% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/photoperiod.R
\name{photoperiod}
\alias{photoperiod}
\alias{extract_photoperiod}
\alias{add_photoperiod}
\alias{solar_noon}
\title{Calculate photoperiod and boundary times}
\usage{
photoperiod(coordinates, dates, tz, solarDep = 6)

extract_photoperiod(
  dataset,
  coordinates,
  Datetime.colname = Datetime,
  solarDep = 6
)

add_photoperiod(
  dataset,
  coordinates,
  Datetime.colname = Datetime,
  solarDep = 6,
  overwrite = FALSE
)

solar_noon(coordinates, dates, tz)
}
\arguments{
\item{coordinates}{A two element numeric vector representing the latitude and
longitude of the location. \emph{Important note: \strong{Latitude} is the first
element and \strong{Longitude} is the second element.}}

\item{dates}{A date of format \code{Date}, or coercible to \code{Date} through
\code{\link[lubridate:as_date]{lubridate::as_date()}}}

\item{tz}{Timezone of the data. Expects a \code{character}. You can look up the
supported timezones with \code{\link[=OlsonNames]{OlsonNames()}}.}

\item{solarDep}{A numerical value representing the solar depression angle
between 90 and -90. This means a value of 6 equals \strong{-6} degrees above the
horizon. Default is 6, equalling \verb{Civil dawn/dusk}. Other common values are
12 degrees for \verb{Nautical dawn/dusk}, 18 degrees for \verb{Astronomical dawn/dusk}, and 0 degrees for \code{Sunrise/Sunset}. Note that the output
columns will always be named \code{dawn} and \code{dusk}, regardless of the
\code{solarDep} value.}

\item{dataset}{A light logger dataset. Expects a \code{dataframe}. If not imported
by \link{LightLogR}, take care to choose a sensible variable for the
\code{Datetime.colname}.}

\item{Datetime.colname}{column name that contains the datetime. Defaults to
\code{"Datetime"} which is automatically correct for data imported with
\link{LightLogR}. Expects a \code{symbol}. Needs to be part of the \code{dataset}. Must
be of type \code{POSIXct}.}

\item{overwrite}{Logical scalar. If \code{TRUE}, the function will overwrite any
columns with the same name. If \code{FALSE} (default), the function will stop if
any of the columns already exist in the dataset.}
}
\value{
\code{\link[=photoperiod]{photoperiod()}} returns a \code{tibble} with the calculated times of dawn
and dusk for the given location and date, with the length equal to the
\code{dates} input parameter . The tibble contains the following columns:
\itemize{
\item \code{date} with the date of the calculation, stored as class \code{Date}
\item \code{tz} with the timezone of the output, stored as class \code{character}
\item \code{lat} and \code{lon} with the latitude and longitude of the location, stored
as class \code{numeric}
\item \code{solar.angle} with the negative solar depression angle, i.e. the sun
elevation above the horizon. stored as class \code{numeric}
\item \code{dawn} and \code{dusk} with the calculated datetimes, stored as class \code{POSIXct}
\item \code{photoperiod} with the calculated photoperiod, stored as class \code{difftime}.
}

\code{\link[=extract_photoperiod]{extract_photoperiod()}} returns a \code{tibble} of the same structure as
\code{\link[=photoperiod]{photoperiod()}}, but with a length equal to the number of unique dates in
the dataset.

\link{add_photoperiod} returns the input dataset with the added
photoperiod information. The information is appended with the following
columns: \code{dawn}, \code{dusk}, \code{photoperiod}, and \code{photoperiod.state}.

\code{\link[=solar_noon]{solar_noon()}} returns a \code{tibble} with the calculated solar noon
}
\description{
A family of functions to extract and add photoperiod information.
\code{\link[=photoperiod]{photoperiod()}} creates a \code{tibble} with the calculated times of dawn and dusk
for the given location and date. The function is a convenience wrapper for
\code{\link[suntools:crepuscule]{suntools::crepuscule()}} to calculate the times of dawn and dusk. By default,
civil dawn and dusk are calculated, but the function can be used to calculate
other times by changing the \code{solarDep} parameter (e.g., 0 for sunrise/sunset,
12 for nautical, and 18 for astronomical).

Taking a light exposure dataset as input, \code{\link[=extract_photoperiod]{extract_photoperiod()}} calculates
the photoperiods and their boundary times for each unique day in the dataset,
given a location and boundary condition (i.e., the solar depression angle).
Basically, this is a convenience wrapper for \code{\link[=photoperiod]{photoperiod()}} that takes a
light logger dataset and extracts unique dates and the time zone from the
dataset.

\code{\link[=add_photoperiod]{add_photoperiod()}} adds photoperiod information to a light logger dataset.
Beyond the photoperiod information, it will categorize the
\code{photoperiod.state} as \code{"day"} or \code{"night"}. If \code{overwrite} is set to \code{TRUE},
the function will overwrite any columns with the same name.

\code{\link[=solar_noon]{solar_noon()}} calculates the solar noon for a given location and date. The
function is a convenience wrapper for \code{\link[suntools:solarnoon]{suntools::solarnoon()}}. The function
has no companions like \code{\link[=extract_photoperiod]{extract_photoperiod()}} or \code{\link[=add_photoperiod]{add_photoperiod()}}, but
will be extended, if there is sufficient interest.
}
\details{
Please note that all functions of the \code{photoperiod} family work with one
coordinate pair at a time. If you have multiple locations (and multiple time
zones), you need to run the function for each location separately. We suggest
using a nested dataframe structure, and employ the \code{purrr} package to iterate
over the locations.
}
\examples{
 #example für Tübingen, Germany
 coordinates <- c(48.521637, 9.057645)
 dates <- c("2023-06-01", "2025-08-23")
 tz <- "Europe/Berlin"
 
 #civil dawn/dusk
 photoperiod(coordinates, dates, tz)
 #sunrise/sunset
 photoperiod(coordinates, dates, tz, solarDep = 0)
 #extract_photoperiod
 sample.data.environment |> 
   extract_photoperiod(coordinates)


#add_photoperiod
added_photoperiod <-
 sample.data.environment |>
 add_photoperiod(coordinates)

added_photoperiod |> head()

added_photoperiod |>
  filter_Date(length = "3 days") |>
  gg_days(aes_col = photoperiod.state,
          group = dplyr::consecutive_id(photoperiod.state),
          jco_color = TRUE
          )

added_photoperiod |>
  filter_Date(length = "3 days") |>
  gg_day(aes_col = Id) +
  ggplot2:: geom_rect(
  data = \(x) x |> dplyr::ungroup(Id) |> dplyr::summarize(dawn = mean(dawn) |> hms::as_hms()),
  ggplot2::aes(xmin = 0, xmax = dawn, ymin = -Inf, ymax = Inf),
  alpha = 0.1
  ) +
  ggplot2:: geom_rect(
  data = \(x) x |> dplyr::ungroup(Id) |> dplyr::summarize(dusk = mean(dusk) |> hms::as_hms()),
  ggplot2::aes(xmin = dusk, xmax = 24*60*60, ymin = -Inf, ymax = Inf),
  alpha = 0.1
  )


 added_photoperiod |> dplyr::summarize(dawn = mean(dawn) |> hms::as_hms())

 #solar_noon()
 solar_noon(coordinates, dates, tz)

}
\seealso{
Other photoperiod: 
\code{\link{gg_photoperiod}()}
}
\concept{photoperiod}
