#'
#' Feature Normalization
#'
#' @description
#' This function normalizes a matrix of features using precomputed means and standard deviations.
#' The function automatically runs \code{\link[LSTMfactors]{load.scaler}}
#' to read the standard deviations and means of the features,
#' which are organized into a \code{list} object named \code{\link[LSTMfactors]{data.scaler.LSTM}}.
#' These means and standard deviations are computed from
#' the 1,000,000 datasets \code{\link[LSTMfactors]{data.datasets.LSTM}} for
#' training the pre-trained Long Short Term Memory (LSTM) Network.
#'
#' @param features A numeric matrix where each row represents an observation and each column represents a feature.
#'
#' @details
#' The function applies z-score normalization to each element in the \code{features} matrix. It uses
#' the \code{scaler} object, which is expected to contain precomputed means and standard deviations for each feature.
#' The normalized value for each element is computed as:
#' \deqn{z = \frac{x - \mu}{\sigma}}
#' where \eqn{x} is the original value, \eqn{\mu} is the mean, and \eqn{\sigma} is the standard deviation.
#'
#' @seealso \code{\link[LSTMfactors]{LSTM}}, \code{\link[LSTMfactors]{load.scaler}}, \code{\link[LSTMfactors]{data.datasets.LSTM}}, \code{\link[LSTMfactors]{data.scaler.LSTM}}
#'
#' @return A matrix of the same dimensions as \code{features}, where each feature has been normalized.
#'
#' @examples
#' library(LSTMfactors)
#' set.seed(123)
#'
#' ##Take the data.DAPCS dataset as an example.
#' data(data.DAPCS)
#'
#' response <- as.matrix(data.DAPCS[, 3:22]) ## loading data
#'
#' \donttest{
#' ## Run extractor.feature function
#' features <- extractor.feature(response)
#'
#' features.nor <- normalizor(features)
#' print(features.nor)
#' }
#'
#'
#'
#' @export
normalizor <- function(features){

  data.scaler <- load.scaler()

  means <- matrix(data.scaler$means, nrow = nrow(features), ncol = ncol(features), byrow=TRUE)

  sds <- matrix(data.scaler$sds, nrow = nrow(features), ncol = ncol(features), byrow=TRUE)

  features.normalized <- (features - means) / sds

  return(features.normalized)
}
