% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KFS.R
\name{KFS}
\alias{KFS}
\title{Kalman Filter and Smoother with Exact Diffuse Initialization for Exponential
Family State Space Models}
\usage{
KFS(
  model,
  filtering,
  smoothing,
  simplify = TRUE,
  transform = c("ldl", "augment"),
  nsim = 0,
  theta,
  maxiter = 50,
  convtol = 1e-08,
  return_model = TRUE,
  expected = FALSE,
  H_tol = 1e+15,
  transform_tol
)
}
\arguments{
\item{model}{Object of class \code{SSModel}.}

\item{filtering}{Types of filtering. Possible choices are \code{"state"},
\code{"signal"}, \code{"mean"}, and \code{"none"}. Default is
\code{"state"} for Gaussian and \code{"none"} for non-Gaussian models.
Multiple values are allowed. For Gaussian models, the signal is the mean.
Note that filtering for non-Gaussian models with importance sampling can be
very slow with large models.}

\item{smoothing}{Types of smoothing. Possible choices are \code{"state"},
\code{"signal"}, \code{"mean"}, \code{"disturbance"}, and \code{"none"}. Default is \code{"state"} and \code{"mean"}. For
non-Gaussian models, option \code{"disturbance"} is not supported, and for
Gaussian models option \code{"mean"} is identical to \code{"signal"}. Multiple values are
allowed.}

\item{simplify}{If \code{FALSE} and the model is completely Gaussian, \code{KFS} returns some
generally not so interesting variables from filtering and smoothing. Default
is \code{TRUE}.}

\item{transform}{How to transform the model in case of non-diagonal
covariance matrix \code{H}. Defaults to \code{"ldl"}. See function
\code{\link{transformSSM}} for details.}

\item{nsim}{The number of independent samples used in importance sampling.
Only used for non-Gaussian models. Default is 0, which computes the
approximating Gaussian model by \code{\link{approxSSM}} and performs the
usual Gaussian filtering/smoothing so that the smoothed state estimates
equals to the conditional mode of \eqn{p(\alpha_t|y)}{p(\alpha[t]|y)}.
In case of \code{nsim = 0}, the mean estimates and their variances are computed using
the Delta method (ignoring the covariance terms).}

\item{theta}{Initial values for conditional mode theta. Only used for
non-Gaussian models.}

\item{maxiter}{The maximum number of iterations used in Gaussian
approximation. Default is 50. Only used for non-Gaussian models.}

\item{convtol}{Tolerance parameter for convergence checks for Gaussian
approximation. Only used for non-Gaussian models.}

\item{return_model}{Logical, indicating whether the original input model should be 
returned as part of the output. Defaults to TRUE, but for large models can be set 
to FALSE in order to save memory. However, many of the methods operating on the 
output of \code{KFS} use this model so this will not work if \code{return_model=FALSE}.}

\item{expected}{Logical value defining the approximation of H_t in case of Gamma 
and negative binomial distribution. Default is \code{FALSE} which matches the 
algorithm of Durbin & Koopman (1997), whereas \code{TRUE} uses the expected value
of observations in the equations, leading to results which match with \code{glm} (where applicable).
The latter case was the default behaviour of KFAS before version 1.3.8.
Essentially this is the difference between observed and expected information in the GLM context. 
Only used for non-Gaussian model.}

\item{H_tol}{Tolerance parameter for check \code{max(H) > tol_H}, which suggests that the approximation 
converged to degenerate case with near zero signal-to-noise ratio. Default is very generous 1e15. 
Only used for non-Gaussian model.}

\item{transform_tol}{Tolerance parameter for LDL decomposition in case of a 
non-diagonal H and \code{transform = "ldl"}. See \code{\link{transformSSM}} and 
\code{\link{ldl}} for details.}
}
\value{
What \code{KFS} returns depends on the arguments \code{filtering},
  \code{smoothing} and \code{simplify}, and whether the model is Gaussian or
  not:

  \item{model}{Original state space model. }

  \item{KFS_transform}{How the non-diagonal \code{H} was handled. }

  \item{logLik}{Value of the log-likelihood function. Only returned for fully
  Gaussian models. }

  \item{a}{One-step-ahead predictions of states, \eqn{a_t = E(\alpha_t | y_{t-1},
  \ldots, y_{1})}{a[t] = E(\alpha[t] | y[t-1], \ldots, y[1])}. }

  \item{P}{Non-diffuse parts of the error covariance matrix of predicted states,
  \eqn{P_t = Var(\alpha_t | y_{t-1}, \ldots, y_{1})
  }{P[t] = Var(\alpha[t] | y[t-1], \ldots, y[1])}. }

  \item{Pinf}{Diffuse part of the error covariance matrix of predicted states.
  Only returned for Gaussian models. }

  \item{att}{Filtered estimates of states, \eqn{a_tt = E(\alpha_t | y_{t},
  \ldots, y_{1})}{a[t] = E(\alpha[t] | y[t], \ldots, y[1])}. }

  \item{Ptt}{Non-diffuse parts of the error covariance matrix of filtered states,
  \eqn{P_tt = Var(\alpha_t | y_{t}, \ldots, y_{1})
  }{P[t] = Var(\alpha[t] | y[t], \ldots, y[1])}. }
  
  \item{t}{One-step-ahead predictions of signals, \eqn{E(Z_t\alpha_t | y_{t-1},
  \ldots, y_{1})}{E(Z[t]\alpha[t] | y[t-1], \ldots, y[1])}. }

  \item{P_theta}{Non-diffuse part of \eqn{Var(Z_t\alpha_t | y_{t-1}, \ldots,
  y_{1})}{Var(Z[t]\alpha[t] | y[t-1], \ldots, y[1])}. }

  \item{m}{One-step-ahead predictions \eqn{f(\theta_t) | y_{t-1}, \ldots,
  y_{1})}{f(\theta[t]) | y[t-1], \ldots, y[1])}, where \eqn{f} is the
  inverse link function. In case of Poisson distribution these predictions are
  multiplied with exposure \eqn{u_t}{u[t]}.  }

  \item{P_mu}{Non-diffuse part of \eqn{Var(f(\theta_t) |
  y_{t-1}, \ldots, y_{1})}{Var(f(\theta[t]) | y[t-1], \ldots, y[1])}.
  In case of Poisson distribution this is \eqn{Var(u_t f(\theta_t) | y_{t-1},
  \ldots, y_{1})}{Var(t[t]f(\theta[t]) | y[t-1], \ldots, y[1])}.
  If \code{nsim = 0}, only diagonal elements (variances) are computed, using the
  Delta method. }

  \item{alphahat}{Smoothed estimates of states, \eqn{E(\alpha_t | y_1, \ldots,
  y_n)}{E(\alpha[t] | y[1], \ldots, y[n])}. }

  \item{V}{Error covariance matrices of smoothed states, \eqn{Var(\alpha_t | y_1,
  \ldots, y_n)}{Var(\alpha[t] | y[1], \ldots, y[n])}. }

  \item{thetahat}{Smoothed estimates of signals, \eqn{E(Z_t\alpha_t | y_1,
  \ldots, y_n)}{E(Z[t]\alpha[t] | y[1], \ldots, y[n])}. }

  \item{V_theta}{Error covariance matrices of smoothed signals
  \eqn{Var(Z[t]\alpha_t | y_1, \ldots, y_n).}{Var(Z[t]\alpha[t]
  | y[1], \ldot , y[n])}. }

  \item{muhat}{Smoothed estimates of \eqn{f(\theta_t) | y_1, \ldots,
  y_n)}{f(\theta[t]) | y[1], \ldots, y[n])}, where \eqn{f} is the inverse
  link function, or in Poisson case \eqn{u_t f(\theta_t) | y_1, \ldots,
  y_n)}{u[t]f(\theta[t]) | y[1], \ldots, y[n])}, where \eqn{u} is the exposure term. }


  \item{V_mu}{Error covariances \eqn{Cov(f(\theta_t)| y_1, \ldots,
  y_n)}{Cov(f(\theta[t]) | y[1], \ldots, y[n])} (or the covariances of
  \eqn{u_t f(\theta_t)}{u[t]f(\theta[t])} given the data in case of Poisson
  distribution). If \code{nsim = 0}, only diagonal elements (variances) are
  computed, using the Delta method.  }

  \item{etahat}{Smoothed disturbance terms \eqn{E(\eta_t | y_1, \ldots,
  y_n)}{E(\eta[t] | y[1], \ldots, y[n])}. Only for Gaussian models. }

  \item{V_eta}{Error covariances \eqn{Var(\eta_t | y_1, \ldots, y_n)}{Var(\eta[t]
  | y[1], \ldots, y[n])}. Note that for computing auxiliary residuals you shoud use method
  \code{\link{rstandard.KFS}}.}

  \item{epshat}{Smoothed disturbance terms \eqn{E(\epsilon_{t,i} | y_1,
  \ldots, y_n)}{E(\epsilon[t,i] | y[1], \ldots, y[n])}. Note that due to
  the possible diagonalization these are on transformed scale.
  Only for Gaussian models. }

  \item{V_eps}{Diagonal elements of \eqn{Var(\epsilon_{t} | y_1, \ldots,
  y_n)}{Var(\epsilon[t] | y[1], \ldots, y[n])}. Note that due to the
  diagonalization the off-diagonal elements are zero.
  Only for Gaussian models. Note that for computing auxiliary residuals you shoud use method
  \code{\link{rstandard.KFS}}.}

  \item{iterations}{The number of iterations used in linearization of
  non-Gaussian model. }

  \item{v}{Prediction errors \eqn{v_{t,i} = y_{t,i} - Z_{i,t}a_{t,i},
  i = 1, \ldots,p}{v[t,i] = y[t,i] - Z[i,t]a[t,i], i = 1, \ldots, p}, where
  \deqn{a_{t,i} = E(\alpha_t | y_{t,i-1}, \ldots, y_{t,1}, \ldots,
  y_{1,1})}{a[t,i] = E(\alpha[t] | y[t,i-1], \ldots, y[t,1], \ldots, y[1,1])}.
  Only returned for Gaussian models.  }

  \item{F}{Prediction error variances \eqn{Var(v_{t,i})}{Var(v[t,i])}. Only
  returned for Gaussian models. }

  \item{Finf}{Diffuse part of prediction error variances. Only returned for Gaussian
  models. }

  \item{d}{The last time index of diffuse phase, i.e. the non-diffuse
  phase began at time \eqn{d+1}. Only returned for Gaussian models.  }


  \item{j}{The last observation index \eqn{i} of \eqn{y_{i,t}}{y[i,t]} of the
  diffuse phase. Only returned for Gaussian models.  }

  In addition, if argument \code{simplify = FALSE}, list contains following
  components:

  \item{K}{Covariances \eqn{Cov(\alpha_{t,i}, y_{t,i} | y_{t,i-1}, \ldots,
  y_{t,1}, y_{t-1}, \ldots , y_{1}), \quad i = 1, \ldots, p}{Cov(\alpha[t,i],
  y[t,i] | y[t,i-1], \ldots, y[t,1], y[t-1], \ldots, y[1]), i = 1, \ldots, p}.
  }

  \item{Kinf}{Diffuse part of \eqn{K_t}{K[t]}.  }

  \item{r}{Weighted sums of innovations \eqn{v_{t+1}, \ldots, v_{n}}{v[t+1],
  \ldots, v[n]}.  Notice that in literature \eqn{t} in \eqn{r_t}{r[t]} goes from
  \eqn{0, \ldots, n}. Here \eqn{t = 1, \ldots, n + 1}. Same applies to all \eqn{r} and
  \eqn{N} variables.  }

  \item{r0, r1}{Diffuse phase decomposition of \eqn{r_t}{r[t]}.  }

  \item{N}{Covariances \eqn{Var(r_t)}{Var(r[t])}.  }

  \item{N0, N1, N2}{Diffuse phase decomposition of \eqn{N_t}{N[t]}.   }
}
\description{
Performs Kalman filtering and smoothing with exact diffuse initialization
using univariate approach for exponential family state space models.
}
\details{
Notice that in case of multivariate Gaussian observations, \code{v}, \code{F},
\code{Finf}, \code{K} and \code{Kinf} are usually not the same as those
calculated in usual multivariate Kalman filter. As filtering is done one
observation element at the time, the elements of the prediction error
\eqn{v_t}{v[t]} are uncorrelated, and \code{F}, \code{Finf}, \code{K} and
\code{Kinf} contain only the diagonal elemens of the corresponding covariance
matrices. The usual multivariate versions of \code{F} and \code{v} can be
obtained from the output of \code{KFS} using the function
\code{\link{mvInnovations}}.

In rare cases (typically with regression components with high multicollinearity or 
long cyclic patterns), the cumulative rounding errors in Kalman filtering and 
smoothing can cause the diffuse phase end too early,
or the backward smoothing gives negative variances (in diffuse and nondiffuse cases). 
Since version 1.4.0, filtering and smoothing algorithms truncate these values to zero during the 
recursions, but this can still leads some numerical problems.
In these cases, redefining the  prior state variances more informative is often helpful. 
Changing the tolerance parameter \code{tol} of the model (see \code{\link{SSModel}}) to smaller 
(or larger), or scaling the model input can sometimes help as well. These numerical issues 
are well known in Kalman filtering/smoothing in general 
(there are other numerically more stable versions available, but these are in general slower).

Fon non-Gaussian models the components corresponding to diffuse filtering
(\code{Finf}, \code{Pinf}, \code{d}, \code{Kinf}) are not returned even
when \code{filtering} is used. Results based on approximating Gaussian model
can be obtained by running \code{KFS} using the output of \code{approxSSM}.

In case of non-Gaussian models with \code{nsim = 0}, the smoothed estimates
relate to the conditional mode of \eqn{p(\alpha|y)}. When using importance
sampling (\code{nsim>0}), results correspond to the conditional mean of
 \eqn{p(\alpha|y)}.
}
\examples{
set.seed(1)
x <- cumsum(rnorm(100, 0, 0.1))
y <- rnorm(100, x, 0.1)
model <- SSModel(y ~ SSMtrend(1, Q = 0.01), H = 0.01)
out <- KFS(model)
ts.plot(ts(x), out$a, out$att, out$alpha, col = 1:4)

}
\references{
Koopman, S.J. and Durbin J. (2000).  Fast filtering and
smoothing for non-stationary time series models, Journal of American
Statistical Assosiation, 92, 1630-38.  \cr

Koopman, S.J. and Durbin J. (2001).  Time Series Analysis by State Space
Methods. Oxford: Oxford University Press.  \cr

Koopman, S.J. and Durbin J. (2003).  Filtering and smoothing of state vector
for diffuse state space models, Journal of Time Series Analysis, Vol. 24,
No. 1.  \cr
}
\seealso{
\code{\link{KFAS}} for examples

\code{\link{logLik}}, \code{\link{KFAS}}, \code{\link{fitSSM}}, 
\code{\link{boat}}, \code{\link{sexratio}},
\code{\link{GlobalTemp}}, \code{\link{SSModel}}, 
\code{\link{importanceSSM}}, \code{\link{approxSSM}} for examples.
}
