localdecodeHSMM <- function(x, M = NA, HSMM, obsdist, dwelldist){

  # Input validation
  if(!dwelldist %in% c("pois", "nbinom", "betabinom")) {
    stop("dwell distribution is not supported")
  }
  if(!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("observation distribution is not supported")
  }

  # Determining if the dwell distribution is shifted
  if(all(HSMM$dwellparameters$shift == 1)){
    shift=FALSE
  } else {
    shift=TRUE
  }

  # Extract model parameters
  J <- length(HSMM$delta)
  obspar <- HSMM$observationparameters
  dwellpar <- HSMM$dwellparameters
  Pi <- HSMM$Pi

  # Use stationary distribution if delta not available
  delta <- solve(t(diag(J) - Pi + 1), rep(1, J))

  # Initialize shift parameters
  if(!isTRUE(shift)){
    dwellpar$shift <- rep(1, J)
  }

  # Setup algorithm parameters
  tau <- length(x)
  if(is.na(M)){
    M <- min(1000, tau)  # Set maximum dwell time
  }

  # Compute dwell-time probabilities and survival functions
  dwell<- dwellprobs(M, J, dwelldist, dwellpar)
  dwellprobs<- dwell$dwellprobs
  surv<- dwell$surv

  # Compute observation probabilities
  obsprobs<- obsprobs(x, J, obsdist, obspar)

  # HSMM forward-backward algorithm
  # Computes posterior state probabilities accounting for dwell times
  backwards<- backwards(
    transProb = as.matrix(Pi),
    initDist = as.numeric(delta),
    obsProb = as.matrix(obsprobs),
    durProb = as.matrix(dwellprobs),  # Dwell-time probabilities
    survProb = as.matrix(surv),       # Survival probabilities
    seqLen = as.integer(tau),
    numStates = as.integer(J),
    maxDur = as.integer(rep(M,J)),
    backwardProb = matrix(0, nrow = tau, ncol=J),
    normConst = numeric(tau),
    eta = numeric(M*J),
    forwardVars = matrix(0, nrow=tau, ncol=J),
    stateProb = numeric(J*tau),
    xi = matrix(0, nrow=tau, ncol=J),    # Posterior state probabilities
    numSeq = as.integer(1),
    totalLen = as.integer(tau),
    backwardVars = matrix(0, nrow=tau, ncol=J)
  )

  # Extract smoothed state probabilities
  smoothedprobs<- as.vector(backwards$xi)

  # Local decoding: most likely state at each time point
  states <- apply(matrix(smoothedprobs, ncol=J), 1, which.max)

  return(states)
}
