% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/barycenter_test.R
\name{ot_barycenter_test}
\alias{ot_barycenter_test}
\title{Test equality of probability vectors}
\usage{
ot_barycenter_test(
  samples,
  costm,
  null.mu = NULL,
  w = NULL,
  num.sim = 1000,
  solver = ot_test_lp_solver(),
  is.metric = is_metric_cost_mat(costm, tol.ti = Inf),
  verbose = FALSE
)
}
\arguments{
\item{samples}{matrix (row-wise) or nested list containing \eqn{K} count vectors. A count vector is a vector of length \eqn{N}
that contains the number of times a sample was observed at the respective points.}

\item{costm}{semi-metric cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.}

\item{null.mu}{probability measures \eqn{\mu} underlying the null distribution. Must be of the same structure as \code{samples}.}

\item{w}{weight vector \eqn{w \in \mathbb{R}^K_+}.}

\item{num.sim}{number of samples to draw from the limiting null distribution.}

\item{solver}{the LP solver to use, see \code{\link{ot_test_lp_solver}}.}

\item{is.metric}{value indicating whether \eqn{c} is a metric cost matrix, see \code{\link{is_metric_cost_mat}}.}

\item{verbose}{logical value indicating whether additional information should be printed.}
}
\value{
An object of class \code{"ot_barycenter_test"} containing:
\tabular{ll}{
\code{mu}           \tab empirical version of \eqn{\mu} that is based on \code{samples} \cr
\code{n}            \tab the sample sizes \cr
\code{p.value}      \tab the \eqn{p}-value \cr
\code{statistic}    \tab the value of the test statistic \cr
\code{null.samples} \tab samples drawn from the null distribution \cr
}
}
\description{
Perform optimal transport (OT) barycenter based tests for equality of probability vectors in a one-way layout.
}
\details{
Denote with \eqn{\mu^1, \ldots, \mu^K} the probability measures that underlie the samples contained in \code{samples}. To test for
the one-way null hypothesis \eqn{H_0 : \mu^1 = \ldots = \mu^K}, this test employs the OT barycenter statistic
which is defined as
\eqn{
 T^B(\mu) := \sqrt{\rho_n} B_c^w(\mu^1, \ldots, \mu^K)\,,
}
where \eqn{\rho_n} is a scaling factor and \eqn{B_c^w} is the OT barycenter functional, see \code{\link{ot_barycenter}}.

The test is based on the asymptotic distribution of \eqn{T^B} under under the null, for more details see the reference.

These simulations can be done in parallel via \code{\link[future:plan]{future::plan}} and the progress can be shown with \code{\link[progressr:with_progress]{progressr::with_progress}}.

Especially for large \eqn{N} and \eqn{K}, simulating a sufficient number of samples from the limiting null distribution might take a while.
Consider using \code{\link{FDOTT}} instead.
}
\examples{

# enable txt progressbar
progressr::handlers("txtprogressbar")
# enable parallel computation
if (requireNamespace("future")) {
    future::plan(future::multisession)
}

K <- 3
N <- 2
costm <- cost_matrix_lp(1:N)

# use higher number to better approximate null distribution and get more accurate p-value
num.sim <- 10

n <- c(300, 360, 200)

# underlying probability vectors
mu <- matrix(1 / N, K, N, TRUE)

# to run this, a LP solver must be available for ROI (ROI.plugin.glpk by default)
if (requireNamespace("ROI.plugin.glpk")) {
    solver <- ot_test_lp_solver("glpk")
    set.seed(123)
    samples <- tab_sample(n, mu)
    progressr::with_progress({
        res <- ot_barycenter_test(samples, costm, num.sim = num.sim, solver = solver)
    })
    print(res)
}

# measures are not equal anymore
mu[2, ] <- 1:N / sum(1:N)

if (requireNamespace("ROI.plugin.glpk")) {
    solver <- ot_test_lp_solver("glpk")
    set.seed(123)
    samples <- tab_sample(n, mu)
    progressr::with_progress({
        res2 <- ot_barycenter_test(samples, costm, num.sim = num.sim, solver = solver)
    })
    print(res2)
}
\dontshow{
## R CMD check: make sure any open connections are closed afterward
if (requireNamespace("future") && !inherits(future::plan(), "sequential")) future::plan(future::sequential)
}
}
\references{
TODO
}
\seealso{
\code{\link{FDOTT}} with \code{H0 = "="}.
}
