% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base_optim_function.R
\name{base.optim}
\alias{base.optim}
\title{Fit Parameters of a Model}
\usage{
base.optim(
  binary,
  parms,
  fit.fn,
  homedir,
  use.optim = TRUE,
  optim.runs = 1,
  default.val = NULL,
  random.borders = 1,
  con.tol = 0.01,
  control.optim = list(maxit = 1000),
  parscale.pars = FALSE,
  scaling = NULL,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{binary}{A vector containing zeroes and ones. Zero indicates that the corresponding parameter is not fitted.}

\item{parms}{A named vector containing the initial values for \code{\link{optim}}. Must be in the same order as \code{binary}.}

\item{fit.fn}{A cost function. Has to take the complete parameter vector as an input argument (needs to be named \code{parms}) and must return must return a selection criterion value (e.g. AICc or BIC). The binary vector containing the information which parameters are fitted, can also be used by taking \code{binary} as an additional function input argument.}

\item{homedir}{A string giving the directory in which the result folders generated by \code{\link{famos}} are found.}

\item{use.optim}{Logical. If true, the cost function \code{fit.fn} will be fitted via \code{\link{optim}}. If FALSE, the cost function will only be evaluated.}

\item{optim.runs}{The number of times that each model will be optimised. Default to 1. Numbers larger than 1 use random initial conditions (see \code{random.borders}).}

\item{default.val}{A named list containing the values that the non-fitted parameters should take. If NULL, all non-fitted parameters will be set to zero. Default values can be either given by a numeric value or by the name of the corresponding parameter the value should be inherited from (NOTE: In this case the corresponding parameter entry has to contain a numeric value). Default to NULL.}

\item{random.borders}{The ranges from which the random initial parameter conditions for all optim.runs > 1 are sampled. Can be either given as a vector containing the relative deviations for all parameters or as a matrix containing in its first column the lower and in its second column the upper border values. Parameters are uniformly sampled based on \code{\link{runif}}. Default to 1 (100\% deviation of all parameters). Alternatively, functions such as \code{\link{rnorm}}, \code{\link{rchisq}}, etc. can be used if the additional arguments are passed along as well.}

\item{con.tol}{The relative convergence tolerance. \code{famos} will rerun \code{\link{optim}} until the relative improvement between the current and the last fit is less than \code{con.tol}. Default is set to 0.01, meaning the fitting will terminate if the improvement is less than 1\% of the previous value.}

\item{control.optim}{Control parameters passed along to \code{optim}. For more details, see \code{\link{optim}}.}

\item{parscale.pars}{Logical. If TRUE (default), the \code{parscale} option will be used when fitting with \code{\link{optim}}. This is helpful, if the parameter values are on different scales.}

\item{scaling}{Numeric vector determining how newly added model parameters are scaled. Only needed if \code{parscale.pars} is TRUE.}

\item{verbose}{Logical. If TRUE, FAMoS will output all details about the current fitting procedure.}

\item{...}{Additional parameters.}
}
\value{
Saves the results obtained from fitting the corresponding model parameters in the respective files, from which they can be accessed by the main function \code{\link{famos}}.
}
\description{
Given a specific model, \code{base.optim} fits the corresponding parameters and saves the output.
}
\details{
The fitting routine of \code{base.optim} is based on the function \code{\link{optim}}. The number of fitting runs can be specified by the \code{optim.runs} parameter in the \code{\link{famos}} function. Here, the first fitting run takes the parameters supplied in \code{parms} as a starting condition, while all following fitting runs sample new initial sets according to a uniform distribution based on the intervals [\code{parms} - \code{abs}(\code{parms}), \code{parms} + \code{abs}(\code{parms})].
Additionally, each fitting run is based on a \code{while}-loop that compares the outcome of the previous and the current fit. Each fitting run is terminated when the specified convergence tolerance \code{con.tol} is reached.
}
\examples{
#setting data
true.p2 <- 3
true.p5 <- 2
sim.data <- cbind.data.frame(range = 1:10,
                             y = true.p2^2 * (1:10)^2 - exp(true.p5 * (1:10)))

#define initial parameter values and corresponding test function
inits <- c(p1 = 3, p2 = 4, p3 = -2, p4 = 2, p5 = 0)

cost_function <- function(parms, binary, data){
  if(max(abs(parms)) > 5){
    return(NA)
  }
  with(as.list(c(parms)), {
    res <- p1*4 + p2^2*data$range^2 + p3*sin(data$range) + p4*data$range - exp(p5*data$range)
    diff <- sum((res - data$y)^2)

    #calculate AICC
    nr.par <- length(which(binary == 1))
    nr.data <- nrow(data)
    AICC <- diff + 2*nr.par + 2*nr.par*(nr.par + 1)/(nr.data - nr.par -1)

    return(AICC)
  })
}

#create directories if needed
make.directories(tempdir())

#optimise the model parameters
base.optim(binary = c(0,1,1,0,1),
           parms = inits,
           fit.fn = cost_function,
           homedir = tempdir(),
           data = sim.data)

#delete tempdir
unlink(paste0(tempdir(),"/FAMoS-Results"), recursive = TRUE)
}
