#' High-Dimensional Two-Sample Mean Test with Centering Adjustment
#'
#' Conducts a high-dimensional two-sample mean test with centering adjustment.
#' This function is designed for cases where the number of variables \( p \) is larger than the sample sizes \( n \) and \( m \).
#'
#' @param X Matrix representing the first sample (rows are observations, columns are variables).
#' @param Y Matrix representing the second sample (rows are observations, columns are variables).
#' @param order Integer specifying the order of centering adjustment (default is 0).
#' @return A list containing:
#'   \item{statistic}{The test statistic value.}
#'   \item{pvalue}{The p-value of the test.}
#'   \item{Tn}{The adjusted test statistic before centering.}
#'   \item{var}{The estimated variance.}
#'
#' @details This function performs a high-dimensional two-sample mean test by adjusting the test statistic for centering.
#' It uses a modified t-statistic and estimates the variance to handle high-dimensional data.
#'
#' @examples
#' # Example usage:
#' set.seed(123)
#' X <- matrix(rnorm(200), nrow = 10, ncol = 20)  # 10 samples, 20 variables
#' Y <- matrix(rnorm(200, mean = 0.5), nrow = 10, ncol = 20)  # Different mean
#' result <- zwl(X, Y, order = 0)
#' print(result)
#'
#' @export
zwl <- function(X, Y, order = 0)
{
   p <- ncol(X)
   n <- nrow(X)
   m <- nrow(Y)
   mean1 <- matrix(rep(1, n), nrow = 1) %*% X / n #
   mean2 <- matrix(rep(1, m), nrow = 1) %*% Y / m #
   var1 <- matrix(rep(1, n), nrow = 1) %*% scale(X, mean1, F) ^ 2 / (n - 1) #
   var2 <- matrix(rep(1, m), nrow = 1) %*% scale(Y, mean2, F) ^ 2 / (m - 1) #
   Sp <- sqrt(var1 / n + var2 / m) #
   t_sq <- ((mean1 - mean2) / Sp) ^ 2 #
   Tn <-  drop(rep(1, p) %*% t(t_sq) / p) #
   X_centered <- sweep(X, 2, mean1)  #
  Y_centered <- sweep(Y, 2, mean2)  #
   cov_X <- (n - 1) * cov(X_centered)
  cov_Y <- (m - 1) * cov(Y_centered)
    cov_hat <- (cov_X + cov_Y) / (n + m - 2)
var_hat <-  drop(rep(1, p) %*% cov_hat %*%
                     matrix(rep(1, p), ncol = 1) / p ^ 2) #
compute_center <- function(data, n, m, ntoorderminus = 0) {
  if (!is.matrix(data)) {
    data <- as.matrix(data)}
  p <- ncol(data)
  total <- n + m
  if (nrow(data) != total) {
stop("The number of rows in 'data' does not match the sum of 'n' and 'm'.")}
  mean_X <- colMeans(data[1:n, , drop = FALSE])
  mean_Y <- colMeans(data[(n + 1):(n + m), , drop = FALSE])
  if (ntoorderminus == 0) {
    center_est <- mean_X - mean_Y} else {center_est <- rep(0, p)    }
  return(center_est)}
center.est <-apply(rbind(X, Y),2,compute_center,n = nrow(X),m = nrow(Y), ntoorderminus = order )
   test_stat <- (Tn - drop(rep(1, p) %*% center.est) / p) / sqrt(var_hat)#   
pvalue <-  2 * (1 - stats::pnorm(abs(test_stat), 0, 1))#
   list(
      statistic = test_stat,
      pvalue = pvalue,
      Tn = Tn,
      var = var_hat
   )
}
