#' High-Dimensional Two-Sample Mean Test
#'
#' Conducts a high-dimensional two-sample mean test with optional variable filtering.
#' This function performs both non-studentized and studentized tests to determine
#' whether the means of two groups are significantly different.
#'
#' @param X Matrix representing the first group of data (variables in rows, observations in columns).
#' @param Y Matrix representing the second group of data (variables in rows, observations in columns).
#' @param m Number of bootstrap samples used for the test (default is 2500).
#' @param filter Logical parameter indicating whether to filter variables based on mean differences (default is TRUE).
#' @param alpha Significance level for the test (default is 0.05).
#'
#' @return A list containing the results of the non-studentized and studentized tests.
#' Each result includes the test statistic, p-value, and method description.
#'
#' @examples
#' # Example usage:
#' X <- matrix(rnorm(1000), nrow = 100, ncol = 10)  # 100 variables, 10 observations
#' Y <- matrix(rnorm(1000, mean = 0.5), nrow = 100, ncol = 10)  # Different mean
#' result <- CZZZ(X, Y, m = 1000, filter = TRUE, alpha = 0.05)
#' print(result)
#'
#' @note This function assumes that the input matrices have the same number of rows (variables).
#' If the number of variables is very large, the computation may be time-consuming.
#'
#' @export
CZZZ= function(X, Y, m = 2500, filter = TRUE, alpha = 0.05) {
  p = nrow(X)
  pn = 1:p
  n1 = ncol(X)
  n2 = ncol(Y)
  if (filter) {
    e = 0.1
    tau1 = 0.1*(2*log(p))^(1/2-e)
    tau2 = (sqrt(2)+sqrt(2)/(2*log(p))+sqrt(2*log(1/alpha)/log(p)))*sqrt(log(p))
    Dk = abs(rowMeans(X) - rowMeans(Y))/sqrt(apply(X,1,var)/n1+apply(Y,1,var)/n2)
    pn = which(Dk >= min(tau1,tau2))
    X = X[pn,,drop = FALSE]
    Y = Y[pn,,drop = FALSE]
    p = nrow(X)
    n1 = ncol(X)
    n2 = ncol(Y)
  }
  Tn = matrix(0, 1, 2)
  if (length(pn)>1) {
       Xm = X - rowMeans(X)
      S1 = Xm %*% t(Xm) / n1
      Ym = Y - rowMeans(Y)
      S2 = Ym %*% t(Ym) / n2
    Sig1 = S1+n1/n2*S2
    Ds = diag(1/sqrt(diag(Sig1)))
    Sig2 = Ds %*% Sig1 %*% Ds
    Z3 = MASS::mvrnorm(m,rep(0,p),Sig1)
    Z4 = MASS::mvrnorm(m,rep(0,p),Sig2)
    #     z3a = quantile(apply(abs(Z3),1,max),alpha)
    #     z4a = quantile(apply(abs(Z4),1,max),alpha)
    Dt = abs(rowMeans(X) - rowMeans(Y))
    sn = diag(Sig1)/n1
    T1 = max(sqrt(n1)*Dt)  #
    T2 = max(Dt/sqrt(sn))   #
    # z3a = quantile(apply(abs(Z3),1,max),alpha)
    # z4a = quantile(apply(abs(Z4),1,max),alpha)
    z3.seq = apply(abs(Z3),1,max)
    z3a = quantile(z3.seq,alpha)
    z4.seq = apply(abs(Z4),1,max)
    z4a = quantile(z4.seq,alpha)
    Tn = c(T1>z3a,T2>z4a)
    names(Tn) = c("Non-Studentized", "Studentized")
    names(T1) = "Non-Studentized Statistic"
    ns.p = sum(T1<z3.seq)/length(z3.seq) #
    ns.res = list(statistics = T1, p.value = ns.p, alternative = "two.sided",
                  method = "Two-Sample HD Non-Studentized test")
    names(T2) = "Studentized Statistic"
    s.p = sum(T2<z4.seq)/length(z4.seq) #
    s.res = list(statistics = T2, p.value = s.p, alternative = "two.sided",
                 method = "Two-Sample HD Studentized test")
    res = list(NonStudent = ns.res, Student = s.res)
    return(res)
  } 
}