create_interactive_plots <- function(data, plot_type = "scatter") {
  
  if (!requireNamespace("plotly", quietly = TRUE)) {
    stop("Package plotly is required but not installed.")
  }
  
  if (plot_type == "scatter") {
    # Interactive scatter plot of sentiment vs location
    p <- ggplot2::ggplot(data, ggplot2::aes(x = Longitude, y = Latitude, 
                                           color = sentiment_category,
                                           text = paste("City:", City,
                                                       "\nSentiment:", sentiment_category,
                                                       "\nScore:", round(sentiment_score, 2),
                                                       "\nComment:", substr(User_Comment, 1, 50), "..."))) +
      ggplot2::geom_point(alpha = 0.7, size = 3) +
      ggplot2::scale_color_manual(values = c("Negative" = "red", "Neutral" = "blue", "Positive" = "green")) +
      ggplot2::theme_minimal() +
      ggplot2::labs(title = "Interactive Geographic Sentiment Distribution")
    
    return(plotly::ggplotly(p, tooltip = "text"))
    
  } else if (plot_type == "bar") {
    # Interactive bar chart
    sentiment_summary <- aggregate(sentiment_score ~ City + sentiment_category, data, length)
    names(sentiment_summary)[3] <- "count"
    
    p <- plotly::plot_ly(sentiment_summary, 
                        x = ~City, 
                        y = ~count, 
                        color = ~sentiment_category,
                        colors = c("red", "blue", "green"),
                        type = "bar") %>%
      plotly::layout(title = "Sentiment Count by City",
                    xaxis = list(title = "City"),
                    yaxis = list(title = "Count"),
                    barmode = "stack")
    
    return(p)
    
  } else if (plot_type == "timeline") {
    # Convert timestamp to Date if it exists
    if ("Timestamp" %in% names(data)) {
      data$Date <- as.Date(data$Timestamp)
      daily_sentiment <- aggregate(sentiment_score ~ Date, data, mean)
      
      p <- plotly::plot_ly(daily_sentiment, 
                          x = ~Date, 
                          y = ~sentiment_score,
                          type = "scatter",
                          mode = "lines+markers",
                          line = list(color = "blue")) %>%
        plotly::layout(title = "Sentiment Trend Over Time",
                      xaxis = list(title = "Date"),
                      yaxis = list(title = "Average Sentiment Score"))
      
      return(p)
    } else {
      stop("Timestamp column not found in data")
    }
  }
}