





#' The probability mass function of the discrete tempered stable distribution.
#'
#' @param x vector of points.
#' @param alpha Index of stability; Number in (0,1)
#' @param eta A parameter, eta>0.
#' @param tp A vector of tempering parameters.
#' @param tf Tempering function. It can be one of the "discrete-stable", "discrete-truncated-stable",
#' "discrete-pRDTS", "poisson-tweedie", "exp-tempering", "beta-prime-tempering", "Pareto-tempering".
#' @param zt Logical. If True it calculates zero-truncated probabilities.
#'
#' @return A vector of probabilities.
#' @export
#'
#' @references M. Grabchak. Discrete tempered stable distributions. Methodology and Computing in Applied Probability, 24(3):1877-1890, 2021.
#' @examples
#' x <- 0:10
#' ddts(x, 0.5, 1)
ddts <- function(x, alpha, eta, tp = c(1,1), tf = "poisson-tweedie", zt = FALSE) {
  valid_types <- c("discrete-stable",
                   "discrete-truncated-stable",
                   "discrete-pRDTS",
                   "poisson-tweedie",
                   "exp-tempering",
                   "beta-prime-tempering",
                   "Pareto-tempering")
  if (!is.logical(zt) || length(zt) != 1){
    stop("zt must be a single logical value")
  }
  if (!tf %in% valid_types) {
    stop(paste("Unknown tempering function", tf, "- returning NA"))
  }
  if (alpha <=0 || alpha >=1) {
    stop(paste("alpha should be between 0 and 1"))
  }
  if (eta <= 0) {
    stop(paste("eta should be a positive number"))
  }
  if(zt == FALSE){
    switch(tf,
           "discrete-stable" = dDTS_DiscreteStable(x, alpha, eta),
           "discrete-truncated-stable" = dDTS_DiscreteTruncatedStable(x, alpha, eta, tp[1]),
           "discrete-pRDTS" = dDTS_pRDTS(x, alpha, eta, tp[1], tp[2]),
           "poisson-tweedie" = dDTS_PoissonTweedie(x, alpha, eta, tp[1]),
           "exp-tempering" = dDTS_Exponential(x, alpha, eta, tp[1]),
           "beta-prime-tempering" = dDTS_BetaPrime(x, alpha, eta, tp[1], tp[2]),
           "Pareto-tempering" = dDTS_Pareto(x, alpha, eta, tp[1]))
  }
  else if(zt == TRUE){
    switch(tf,
           "discrete-stable" = dztDTS_DiscreteStable(x, alpha, eta),
           "discrete-truncated-stable" = dztDTS_DiscreteTruncatedStable(x, alpha, eta, tp[1]),
           "discrete-pRDTS" = dztDTS_pRDTS(x, alpha, eta, tp[1], tp[2]),
           "poisson-tweedie" = dztDTS_PoissonTweedie(x, alpha, eta, tp[1]),
           "exp-tempering" = dztDTS_Exponential(x, alpha, eta, tp[1]),
           "beta-prime-tempering" = dztDTS_BetaPrime(x, alpha, eta, tp[1], tp[2]),
           "Pareto-tempering" = dztDTS_Pareto(x, alpha, eta, tp[1]))
  }
}




#-------------------------------------------------------------------------------


#' The probability distribution of the discrete tempered stable distribution.
#'
#' @param x vector of points.
#' @param alpha Index of stability; Number in (0,1)
#' @param eta A parameter, eta>0.
#' @param tp A vector of tempering parameters.
#' @param tf Tempering function. It can be one of the "discrete-stable", "discrete-truncated-stable",
#' "discrete-pRDTS", "poisson-tweedie", "exp-tempering", "beta-prime-tempering", "Pareto-tempering".
#' @param zt Logical. If True it calculates zero-truncated probabilities.
#'
#' @return A vector of numbers.
#' @export
#'
#' @references M. Grabchak. Discrete tempered stable distributions. Methodology and Computing in Applied Probability, 24(3):1877-1890, 2021.
#' @examples
#' x <- 0:10
#' pdts(x, 0.5, 1)
pdts <- function(x, alpha, eta, tp = c(1,1), tf = "poisson-tweedie", zt = FALSE){
  valid_types <- c("discrete-stable",
                   "discrete-truncated-stable",
                   "discrete-pRDTS",
                   "poisson-tweedie",
                   "exp-tempering",
                   "beta-prime-tempering",
                   "Pareto-tempering")

  if (!is.logical(zt) || length(zt) != 1){
    stop("zt must be a single logical value")
  }
  if (!tf %in% valid_types) {
    stop(paste("Unknown tempering function", tf, "- returning NA"))
  }
  if (alpha <=0 || alpha >=1) {
    stop(paste("alpha must be between 0 and 1"))
  }
  if (eta <= 0) {
    stop(paste("eta must be a positive number"))
  }

  if(zt == FALSE){
    switch(tf,
           "discrete-stable" = pDTS_DiscreteStable(x, alpha, eta),
           "discrete-truncated-stable" = pDTS_DiscreteTruncatedStable(x, alpha, eta, tp[1]),
           "discrete-pRDTS" = pDTS_pRDTS(x, alpha, eta, tp[1], tp[2]),
           "poisson-tweedie" = pDTS_PoissonTweedie(x, alpha, eta, tp[1]),
           "exp-tempering" = pDTS_Exponential(x, alpha, eta, tp[1]),
           "beta-prime-tempering" = pDTS_BetaPrime(x, alpha, eta, tp[1], tp[2]),
           "Pareto-tempering" = pDTS_Pareto(x, alpha, eta, tp[1])
    )
  }
  else if(zt == TRUE){
    switch(tf,
           "discrete-stable" = pztDTS_DiscreteStable(x, alpha, eta),
           "discrete-truncated-stable" = pztDTS_DiscreteTruncatedStable(x, alpha, eta, tp[1]),
           "discrete-pRDTS" = pztDTS_pRDTS(x, alpha, eta, tp[1], tp[2]),
           "poisson-tweedie" = pztDTS_PoissonTweedie(x, alpha, eta, tp[1]),
           "exp-tempering" = pztDTS_Exponential(x, alpha, eta, tp[1]),
           "beta-prime-tempering" = pztDTS_BetaPrime(x, alpha, eta, tp[1], tp[2]),
           "Pareto-tempering" = pztDTS_Pareto(x, alpha, eta, tp[1])
    )
  }
}




#------------------------

#' Simulation from a discrete tempered stable distribution.
#'
#' @param n Number of observations.
#' @param alpha Index of stability; Number in (0,1)
#' @param eta A parameter, eta>0.
#' @param tp A vector of tempering parameters.
#' @param tf Tempering function. It can be one of the "discrete-stable", "discrete-truncated-stable",
#' "discrete-pRDTS", "poisson-tweedie", "exp-tempering", "beta-prime-tempering", "Pareto-tempering".
#' @param c The essential supremum of the tempering function.
#' @param zt Logical. If True it calculates zero-truncated probabilities.
#'
#' @return A vector of observations from a DTS distributions.
#' @export
#'
#' @references M. Grabchak. Discrete tempered stable distributions. Methodology and Computing in Applied Probability, 24(3):1877-1890, 2021.
#' @examples
#' n <- 10
#' rdts(n, 0.5, 1)
rdts <- function(n, alpha, eta, tp = c(1,1), tf = "poisson-tweedie", c = 1, zt = FALSE){
  valid_types <- c("discrete-stable",
                   "discrete-truncated-stable",
                   "discrete-pRDTS",
                   "poisson-tweedie",
                   "exp-tempering",
                   "beta-prime-tempering",
                   "Pareto-tempering")

  if (!is.logical(zt) || length(zt) != 1){
    stop("zt must be a single logical value")
  }
  if (!tf %in% valid_types) {
    warning(paste("Unknown tempering function", tf, "- returning NA"))
    return(NA)
  }
  if (alpha <=0 || alpha >=1) {
    warning(paste("alpha should be between 0 and 1"))
    return(NA)
  }

  if (eta <= 0) {
    warning(paste("eta should be a positive number"))
    return(NA)
  }
  if(zt == FALSE){
    switch(tf,
           "discrete-stable" = rDTS_DiscreteStable(n, alpha, eta, c),
           "discrete-truncated-stable" = rDTS_DiscreteTruncatedStable(n, alpha, eta, tp[1], c),
           "discrete-pRDTS" = rDTS_pRDTS(n, alpha, eta, tp[1], tp[2], c),
           "poisson-tweedie" = rDTS_PoissonTweedie(n, alpha, eta, tp[1], c),
           "exp-tempering" = rDTS_Exponential(n, alpha, eta, tp[1], c),
           "beta-prime-tempering" = rDTS_BetaPrime(n, alpha, eta, tp[1], tp[2], c),
           "Pareto-tempering" = rDTS_Pareto(n, alpha, eta, tp[1], c))
  }
  else if(zt == TRUE){
    switch(tf,
           "discrete-stable" = rztDTS_DiscreteStable(n, alpha, eta, c),
           "discrete-truncated-stable" = rztDTS_DiscreteTruncatedStable(n, alpha, eta, tp[1], c),
           "discrete-pRDTS" = rztDTS_pRDTS(n, alpha, eta, tp[1], tp[2], c),
           "poisson-tweedie" = rztDTS_PoissonTweedie(n, alpha, eta, tp[1], c),
           "exp-tempering" = rztDTS_Exponential(n, alpha, eta, tp[1], c),
           "beta-prime-tempering" = rztDTS_BetaPrime(n, alpha, eta, tp[1], tp[2], c),
           "Pareto-tempering" = rztDTS_Pareto(n, alpha, eta, tp[1], c))
  }
}



#-----------------------------------------------------------------------------------------------------------



#' Log-likelihood function for a discrete tempered stable distribution.
#'
#' @param pv A vector of parameters.
#' @param smpl A sample data to be used for estimation.
#' @param tf Tempering function. It can be one of the "discrete-stable", "discrete-truncated-stable",
#' "discrete-pRDTS", "poisson-tweedie", "exp-tempering", "beta-prime-tempering", "Pareto-tempering".
#' @param zt  Logical. If True it calculates zero-truncated probabilities.
#'
#' @return A number. Negative of likelihood function.
#' @export
#' @references M. Grabchak. Discrete tempered stable distributions. Methodology and Computing in Applied Probability, 24(3):1877-1890, 2021.
#' @examples
#' pv <- c(0.5, 1, 1)
#' n <- 100
#' smpl <- rdts(n, 0.5, 1)
#' edts(pv, smpl)
edts <- function(pv, smpl, tf = "poisson-tweedie", zt = FALSE){
  valid_types <- c("discrete-stable",
                   "discrete-truncated-stable",
                   "discrete-pRDTS",
                   "poisson-tweedie",
                   "exp-tempering",
                   "beta-prime-tempering",
                   "Pareto-tempering")
  if (!tf %in% valid_types) {
    warning(paste("Unknown tempering function", tf, "- returning NA"))
    return(NA)
  }

  if(zt == FALSE){
    switch(tf,
           "discrete-stable" = logLik_DiscreteStable(pv, smpl),
           "discrete-truncated-stable" = logLik_DiscreteTruncatedStable(pv, smpl),
           "discrete-pRDTS" =  logLik_pRDTS(pv, smpl),
           "poisson-tweedie" = logLik_PoissonTweedie(pv, smpl),
           "exp-tempering" =  logLik_Exponential(pv, smpl),
           "beta-prime-tempering" =  logLik_BetaPrime(pv, smpl),
           "Pareto-tempering" =  logLik_Pareto(pv, smpl)
    )
  }
  else if(zt == TRUE){
    switch(tf,
           "discrete-stable" = logLik_Tzt_DiscreteStable(pv, smpl),
           "discrete-truncated-stable" = logLik_Tzt_DiscreteTruncatedStable(pv, smpl),
           "discrete-pRDTS" =  logLik_Tzt_pRDTS(pv, smpl),
           "poisson-tweedie" = logLik_Tzt_PoissonTweedie(pv , smpl),
           "exp-tempering" =  logLik_Tzt_Exponential(pv, smpl),
           "beta-prime-tempering" =  logLik_Tzt_BetaPrime(pv, smpl),
           "Pareto-tempering" =  logLik_Tzt_Pareto(pv, smpl)
    )
  }
  else{stop("zt must be TRUE or FALSE")}
}

#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
rztpois_sampler <- function(n, lambda){
  N <- n * 5 * as.integer((lambda)/(1-exp(-lambda))) + 100
  u <- stats::runif(N, 0, 1)
  k <- stats::rpois(N, lambda) + 1
  out <- k[u < 1/k]
  if (length(out) < n) {
    additional_samples <- rztpois_sampler(n*2, lambda)
    out <- c(out, additional_samples)
  }
  return(out[1:n])
}

rztpois_er <- function(n, lambda){
  if(length(n)>1){
    n <- length(n)
  }
  out <- rep(NA, n)
  index <- 1:n
  len_lam <- length(lambda)
  index <- rep(1:len_lam, length.out = n)
  for(i in 1:len_lam){
    out[index == i] <- rztpois_sampler(n = sum(index == i), lambda = lambda[i])
  }
  return(out)
}

#-------------------------------------------------------------------------------
#common functions for DTS

pmf.out <- function(x,p){
  out <- rep(0, length(x))
  for(i in 1:length(x)){
    if(as.integer(x[i]) != x[i]) {
      warning(paste("non-integer x = ", x[i]))
      out[i] <- 0
    }
    else if(x[i] < 0){
      out[i] <- 0
    }
    else if (x[i] >= 0){
      out[i] <- p[x[i]+1]
    }
  }
  return(out)
}

rF1 <- function(n, alpha, eta, l_plus, q.x, c){
  prob_F1 <- (l_plus * alpha * (1-alpha)) /(c)
  N <- round(n * (1.05/ prob_F1))
  u.1 <- stats::runif(N, 0, 1)
  u.2 <- stats::runif(N, 0, 1)
  X <- (u.1^( 1/(1-alpha)) * u.2^(-1/alpha))
  m = X
  m[m>1] = 1
  phi <- (q.x(X) * (1-exp(-X)) )/ (c * m)
  u <- stats::runif(length(phi),0,1)
  lam <- X[u <= phi]
  if(length(lam)<n){
    additionalSamples <- rF1(n-length(lam) , alpha, eta, l_plus, q.x, c)
    lam <- c(lam , additionalSamples)
  }
  return(lam[1:n])
}

rF2 <- function(n, alpha, eta, l_plus, q.x, c, p, xi){
  prob_F2 <- (xi * ((1+1/xi)^alpha - 1) ) / (alpha)
  N <- round(n * (1.05/ prob_F2)) + 10
  X <- stats::rgamma(n, (1-alpha)/p, xi)
  X <- X^(1/p)
  phi <- (q.x(X) * (1-exp(-X)) )/( X * exp(-xi * X^p))
  u <- stats::runif(length(phi),0,1)
  lam <- X[u <= phi]
  if(length(lam)<n){
    additionalSamples <- rF2(n-length(lam) , alpha, eta, l_plus, q.x, c, p, xi)
    lam <- c(lam , additionalSamples)
  }

  return(lam[1:n])
}

zeroTrunPois <- function(lam){
  Y <- rep(NA, length(lam))

  low_bound <- lam<0.001
  mid_bound <- lam >= 0.001 & lam <= 1000000000
  high_bound <- lam>1000000000

  if(any(low_bound)){
    Y[low_bound] <- rztpois_er(length(lam[low_bound]),mean(lam[low_bound]))
  }
  if(any(mid_bound)){
    Y[mid_bound] <- actuar::rztpois(length(lam[mid_bound]),lam[mid_bound])
  }

  if(any(high_bound)){
    sim_pois <- rep(NA, length(lam[high_bound]))

    for(i in 1:length(lam[high_bound])){
      sample0 <- stats::rpois(10, lam[high_bound][i])
      sim_pois[i] <- sample0[sample0 > 0][1]
    }
    Y[high_bound] <- sim_pois
  }
  return(Y)
}


sumfun <- function(n, Y, N) {
  DTS <- rep(0, n)
  nonzero_indices <- which(N != 0)
  nonZeroN <- N[N != 0]
  cumsum_N <- c(0, cumsum(nonZeroN))
  out <- rep(NA, length(nonZeroN))
  for(i in 1:length(nonZeroN)){
    out[i] <- sum(Y[(cumsum_N[i]+1):(cumsum_N[i+1])])
  }

  DTS[nonzero_indices] <- out
  return(DTS[1:n])
}



#-------------------------------------------------------------------------------
#Discrete Stable

dDTS_DiscreteStable <- function(x, alpha, eta){
  l_plus <- gamma(1-alpha)/alpha
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi = rep(NA, k)
    for(j in 1:k){
      coeffi[j] = beta(j - alpha, alpha)/ gamma(alpha)
    }

    for(i in 1:k){
      p[i+1] = (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}

pDTS_DiscreteStable <- function(x, alpha, eta){
  k <- max(as.integer(x))
  prob <- dDTS_DiscreteStable(-1:k, alpha, eta)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_DiscreteStable <- function(n, alpha, eta, c){
  l_plus <- gamma(1-alpha)/alpha
  q.x <- function(x){
    return(rep(1, length(x)))
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_DiscreteStable <- function(n, alpha, eta, c){
  l_plus <- gamma(1-alpha)/alpha
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_DiscreteStable(sum(N), alpha, eta, c)
  return(sumfun(n, Y, N))
}

dztDTS_DiscreteStable <- function(x, alpha, eta){
  k <- max(as.integer(x))
  prob <- dDTS_DiscreteStable(0:k, alpha, eta)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_DiscreteStable <- function(x, alpha, eta){
  k <- max(as.integer(x))
  prob <- dztDTS_DiscreteStable(0:k, alpha, eta)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_DiscreteStable <- function(n, alpha, eta, c){
  l_plus <- gamma(1-alpha)/alpha
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_DiscreteStable(sum(N), alpha, eta, c)
  return(sumfun(n, Y, N))
}


logLik_DiscreteStable = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(10000000)
  }
  else
  {
    theo = dDTS_DiscreteStable(0:k, c[1], c[2])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] <- sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo[1:(k+1)]))))
  }
}

logLik_Tzt_DiscreteStable = function(c, smpl){

  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo= dDTS_DiscreteStable(0:k, c[1], c[2])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo[2:(k+1)])) - length(smpl) * log(1 - theo[1])))
  }
}



#-------------------------------------------------------------------------------
# Discrete truncated stable

dDTS_DiscreteTruncatedStable = function(x, alpha, eta, a){
  l_plus <- ((gamma(1-alpha) - expint::gammainc(1-alpha, a)) - ( (1-exp(-a)) * a^(-alpha)))/alpha
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi <- rep(NA, k)
    for(j in 1:k){
      coeffi[j] <- (gamma(j-alpha) - expint::gammainc(j-alpha, a)) / gamma(j)
    }
    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}


pDTS_DiscreteTruncatedStable <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dDTS_DiscreteTruncatedStable(-1:k, alpha, eta, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_DiscreteTruncatedStable <- function(n, alpha, eta, a, c){
  l_plus <- ((gamma(1-alpha) - expint::gammainc(1-alpha, a)) - ( (1-exp(-a)) * a^(-alpha)))/alpha
  q.x <- function(x){
    temp <- rep(NA, length(x))
    for(i in 1:length(x)){
      temp[i] <- ifelse((x[i]>= 0 && x[i] < a),1,0)
    }
    return(temp)
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_DiscreteTruncatedStable <- function(n, alpha, eta, a, c){
  l_plus <- ((gamma(1-alpha) - expint::gammainc(1-alpha, a)) - ( (1-exp(-a)) * a^(-alpha)))/alpha
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_DiscreteTruncatedStable(sum(N), alpha, eta, a, c)
  return(sumfun(n, Y, N))
}

dztDTS_DiscreteTruncatedStable <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dDTS_DiscreteTruncatedStable(0:k, alpha, eta, a)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_DiscreteTruncatedStable <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dztDTS_DiscreteTruncatedStable(0:k, alpha, eta, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_DiscreteTruncatedStable <- function(n, alpha, eta, a, c){
  l_plus <- ((gamma(1-alpha) - expint::gammainc(1-alpha, a)) - ( (1-exp(-a)) * a^(-alpha)))/alpha
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_DiscreteTruncatedStable(sum(N), alpha, eta, a, c)
  return(sumfun(n, Y, N))
}



logLik_DiscreteTruncatedStable   = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo = dDTS_DiscreteTruncatedStable (0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo[1:(k+1)]))))
  }
}

logLik_Tzt_DiscreteTruncatedStable = function(c, smpl){

  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo= dDTS_DiscreteTruncatedStable (0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo[2:(k+1)])) - length(smpl) * log(1 - theo[1])))
  }
}

#-------------------------------------------------------------------------------
# pRDTS

dDTS_pRDTS <- function(x, alpha, eta, q, a){
  l_plus <-  stats::integrate(function(y) (1-exp(-y)) * exp(-a * y^q) * y^(-alpha-1), 0, Inf)[1]$value
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.


  if(k >= 1){
    l <- rep(NA, k)
    coeffi <- rep(NA, k)
    for(j in 1:k){
      l[j] <- stats::integrate(function(y) ((exp(-y-(a * y^q) )) * (y^(j-alpha-1))), 0, Inf)[1]$value
      coeffi[j] <- l[j]/gamma(j)
    }
    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}


pDTS_pRDTS <- function(x, alpha, eta, q, a){
  k <- max(as.integer(x))
  prob <- dDTS_pRDTS(-1:k, alpha, eta, q, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}



rL_pRDTS <- function(n, alpha, eta, q, a, c){
  l_plus <-  stats::integrate(function(y) (1-exp(-y)) * exp(-a * y^q) * y^(-alpha-1), 0, Inf)[1]$value
  q.x <- function(x){
    return(exp(-a * x^q))
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_pRDTS <- function(n, alpha, eta, q, a, c){
  l_plus <-  stats::integrate(function(y) (1-exp(-y)) * exp(-a * y^q) * y^(-alpha-1), 0, Inf)[1]$value
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_pRDTS(sum(N), alpha, eta, q, a, c)
  return(sumfun(n, Y, N))
}


dztDTS_pRDTS <- function(x, alpha, eta, q, a){
  k <- max(as.integer(x))
  prob <- dDTS_pRDTS(0:k, alpha, eta, q, a)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_pRDTS <- function(x, alpha, eta, q, a){
  k <- max(as.integer(x))
  prob <- dztDTS_pRDTS(0:k, alpha, eta, q, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_pRDTS<- function(n, alpha, eta, q, a, c){
  l_plus <-  stats::integrate(function(y) (1-exp(-y)) * exp(-a * y^q) * y^(-alpha-1), 0, Inf)[1]$value
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_pRDTS(sum(N), alpha, eta, q, a, c)
  return(sumfun(n, Y, N))
}


logLik_pRDTS = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95|| c[1] < 0.1)
  {
    return(1000000000)
  }
  else
  {
    theo = dDTS_pRDTS(0:k, c[1], c[2], c[3], c[4])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo[1:(k+1)]))))
  }
}

logLik_Tzt_pRDTS = function(c, smpl){

  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo= dDTS_pRDTS(0:k, c[1], c[2], c[3], c[4])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo[2:(k+1)])) - length(smpl) * log(1 - theo[1])))
  }
}


#-------------------------------------------------------------------------------
#Poisson Tweedie
dDTS_PoissonTweedie <- function(x, alpha, eta, a){

  l_plus <-  (gamma(1-alpha)/alpha)*((a+1)^alpha-a^alpha)
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi <- rep(NA, k)
    for(j in 1:k){
      coeffi[j] <- ((1+a)^(-(j-alpha)))*(beta(j-alpha, alpha)/gamma(alpha))
    }
    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}

pDTS_PoissonTweedie <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dDTS_PoissonTweedie(-1:k, alpha, eta, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_PoissonTweedie <- function(n, alpha, eta, a, c){
  l_plus <- (gamma(1-alpha)/alpha)*((a+1)^alpha-a^alpha)
  q.x <- function(x){
    return(exp(-a*x))
  }
  prob_F1 <- (l_plus * alpha * (1-alpha)) /(c)
  prob_F2 <- (a * ((1+1/a)^alpha - 1) ) / (alpha)
  if(prob_F1 >= prob_F2){
    lam <- rF1(n, alpha, eta, l_plus, q.x, c)

    return(zeroTrunPois(lam))
  }
  else{
    lam <- rF2(n, alpha, eta, l_plus, q.x, c, 1, a)
    return(zeroTrunPois(lam))
  }
}

rDTS_PoissonTweedie <- function(n, alpha, eta, a, c){
  l_plus <- (gamma(1-alpha)/alpha)*((a+1)^alpha-a^alpha)
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_PoissonTweedie (sum(N), alpha, eta, a, c)
  return(sumfun(n, Y, N))
}


dztDTS_PoissonTweedie <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dDTS_PoissonTweedie(0:k, alpha, eta, a)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_PoissonTweedie <- function(x, alpha, eta, a){
  k <- max(as.integer(x))
  prob <- dztDTS_PoissonTweedie(0:k, alpha, eta, a)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_PoissonTweedie <- function(n, alpha, eta, a, c){
  l_plus <- (gamma(1-alpha)/alpha)*((a+1)^alpha-a^alpha)
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_PoissonTweedie(sum(N), alpha, eta, a, c)
  return(sumfun(n, Y, N))
}



logLik_PoissonTweedie = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.99 || c[1] < 0.001)
  {
    return(100000000000)
  }
  else
  {
    theo_poissonTweedie = dDTS_PoissonTweedie(0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo_poissonTweedie[1:(k+1)]))))
  }
}

logLik_Tzt_PoissonTweedie = function(c, smpl){

  k <- max(smpl)
  if (c[1] >= 0.99 || c[1] < 0.001)
  {
    return(100000000000)
  }
  else
  {
    theo_poissonTweedie = dDTS_PoissonTweedie(0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo_poissonTweedie[2:(k+1)])) - length(smpl) * log(1 - theo_poissonTweedie[1])))
  }
}


#-------------------------------------------------------------------------------
#Exponential Tempering

dDTS_Exponential = function(x, alpha, eta, lambda){
  l_plus <- gamma(1-alpha) * lambda^(-alpha) * (exp(lambda) * expint::gammainc(alpha+1, lambda) - gamma(alpha + 1))/alpha
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi <- rep(NA, k)
    for(j in 1:k){
      temp <- stats::integrate(function(t) exp(-t) * (t/lambda)^(alpha-j), lower =  lambda, upper = Inf)[1]$value
      coeffi[j] <-  (exp(lambda)* temp * beta(j - alpha, alpha))/gamma(alpha)
    }
    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}

pDTS_Exponential <- function(x, alpha, eta, lambda){
  k <- max(as.integer(x))
  prob <- dDTS_Exponential(-1:k, alpha, eta, lambda)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_Exponential <- function(n, alpha, eta, lambda, c){
  l_plus <- gamma(1-alpha) * lambda^(-alpha) * (exp(lambda) * expint::gammainc(alpha+1, lambda) - gamma(alpha + 1))/alpha
  q.x <- function(x){
    return(lambda/(x+lambda))
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_Exponential <- function(n, alpha, eta, lambda, c){
  l_plus <- gamma(1-alpha) * lambda^(-alpha) * (exp(lambda) * expint::gammainc(alpha+1, lambda) - gamma(alpha + 1))/alpha
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_Exponential(sum(N), alpha, eta, lambda, c)
  return(sumfun(n, Y, N))
}


dztDTS_Exponential <- function(x, alpha, eta, lambda){
  k <- max(as.integer(x))
  prob <- dDTS_Exponential(0:k, alpha, eta, lambda)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_Exponential <- function(x, alpha, eta, lambda){
  k <- max(as.integer(x))
  prob <- dztDTS_Exponential(0:k, alpha, eta, lambda)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_Exponential <- function(n, alpha, eta, lambda, c){
  l_plus <- gamma(1-alpha) * lambda^(-alpha) * (exp(lambda) * expint::gammainc(alpha+1, lambda) - gamma(alpha + 1))/alpha
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_Exponential(sum(N), alpha, eta, lambda, c)
  return(sumfun(n, Y, N))
}



logLik_Exponential  = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo_Exponential = dDTS_Exponential(0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo_Exponential[1:(k+1)]))))
  }
}

logLik_Tzt_Exponential  = function(c, smpl){

  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
  {
    return(100000000000)
  }
  else
  {
    theo_Exponential= dDTS_Exponential(0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo_Exponential[2:(k+1)])) - length(smpl) * log(1 - theo_Exponential[1])))
  }
}


#-------------------------------------------------------------------------------
#Beta Prime Tempering
dDTS_BetaPrime = function(x, alpha, eta, beta, rho){
  l_plus <- ((gamma(1-alpha)/(alpha * beta(rho , beta))) * (beta(rho, beta - alpha) - beta(alpha + rho, beta - alpha)))
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi <- rep(NA, k)
    for(j in 1:k){
      coeffi[j] <- (beta(j-alpha, alpha) * beta(rho, beta + j - alpha)) / (gamma(alpha) * beta(rho, beta))
    }
    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}

pDTS_BetaPrime <- function(x, alpha, eta, beta, rho){
  k <- max(as.integer(x))
  prob <- dDTS_BetaPrime(-1:k, alpha, eta, beta, rho)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_BetaPrime <- function(n, alpha, eta, beta, rho, c){
  l_plus <- ((gamma(1-alpha)/(alpha * beta(rho , beta))) * (beta(rho, beta - alpha) - beta(alpha + rho, beta - alpha)))
  q.x <- function(x){
    temp <- rep(NA, length(x))
    for(i in 1:length(x)){
      temp[i] <- stats::integrate(function(s) exp(-x[i]*s) * s^(rho-1) * (1+s)^(-beta - rho), 0, Inf)[1]$value
    }
    q <- temp/beta(rho, beta)
    return(q)
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_BetaPrime <- function(n, alpha, eta, beta, rho, c){
  l_plus <- ((gamma(1-alpha)/(alpha * beta(rho , beta))) * (beta(rho, beta - alpha) - beta(alpha + rho, beta - alpha)))
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_BetaPrime(sum(N), alpha, eta, beta, rho, c)
  return(sumfun(n, Y, N))
}

dztDTS_BetaPrime <- function(x, alpha, eta, beta, rho){
  k <- max(as.integer(x))
  prob <- dDTS_BetaPrime(0:k, alpha, eta, beta, rho)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_BetaPrime <- function(x, alpha, eta, beta, rho){
  k <- max(as.integer(x))
  prob <- dztDTS_BetaPrime(0:k,alpha, eta, beta, rho)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_BetaPrime <- function(n, alpha, eta, beta, rho, c){
  l_plus <- ((gamma(1-alpha)/(alpha * beta(rho , beta))) * (beta(rho, beta - alpha) - beta(alpha + rho, beta - alpha)))
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_BetaPrime(sum(N), alpha, eta, beta, rho, c)
  return(sumfun(n, Y, N))
}





logLik_BetaPrime = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
    return(100000000000)
  else if(c[3]<= c[1])
    return(100000000000)
  else
  {
    theo_BetaPrime = dDTS_BetaPrime(0:k, c[1], c[2], c[3], c[4])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo_BetaPrime[1:(k+1)]))))
  }
}

logLik_Tzt_BetaPrime = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95|| c[1] < 0.1)
    return(100000000000)
  else if(c[3]<= c[1])
    return(100000000000)
  else
  {
    theo_BetaPrime = dDTS_BetaPrime(0:k, c[1], c[2], c[3], c[4])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo_BetaPrime[2:(k+1)])) - length(smpl) * log(1 - theo_BetaPrime[1])))
  }
}





#-------------------------------------------------------------------------------
#Pareto
dDTS_Pareto <- function(x, alpha, eta, beta){
  l_plus <- ((gamma(1-alpha) * beta ) / alpha ) * ((1/(beta - alpha)) - beta(alpha+1, beta - alpha))
  k <- max(as.integer(x))
  p <- rep(NA, k+1)
  p[1] <- exp(-eta * l_plus)
  # this for loop will calculate the coefficients to be multiplied by probabilities.
  if(k >= 1){
    coeffi = rep(NA, k)
    for(j in 1:k){
      coeffi[j] <- ( beta /(j + beta - alpha) ) * (beta(j-alpha, alpha)/gamma(alpha))
    }

    for(i in 1:k){
      p[i+1] <- (eta/i)*(sum(rev(coeffi[1:i])*p[1:i]))
    }
  }
  return(pmf.out(x,p))
}

pDTS_Pareto <- function(x, alpha, eta, beta){
  k <- max(as.integer(x))
  prob <- dDTS_Pareto(-1:k, alpha, eta, beta)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(0:(k), c_prob)
  return(cdf(x))
}

rL_Pareto <- function(n, alpha, eta, beta, c){
  l_plus <- ((gamma(1-alpha) * beta ) / alpha ) * ((1/(beta - alpha)) - beta(alpha+1, beta - alpha))
  q.x <- function(x){
    q <- rep(0, length(x))
    for(i in 1:length(x)){
      q[i] <- stats::integrate(function(v) beta * exp(-x[i] * v) * (v + 1)^(-beta-1), 0.0001, Inf, rel.tol = 0.00001)[1]$value
    }
    #q <- sapply(x, function(y) { stats::integrate(function(v) beta * exp(-y * v) * (v + 1)^(-beta-1), 0, Inf)[1]$value})
    return(q)
  }
  lam <- rF1(n, alpha, eta, l_plus, q.x, c)
  return(zeroTrunPois(lam))
}

rDTS_Pareto <- function(n, alpha, eta, beta, c){
  l_plus <- ((gamma(1-alpha) * beta ) / alpha ) * ((1/(beta - alpha)) - beta(alpha+1, beta - alpha))
  N <- stats::rpois(n, eta * l_plus)
  Y <- rL_Pareto(sum(N), alpha, eta, beta, c)
  return(sumfun(n, Y, N))
}

dztDTS_Pareto <- function(x, alpha, eta, beta){
  k <- max(as.integer(x))
  prob <- dDTS_Pareto(0:k, alpha, eta, beta)
  p <- rep(NA, length(prob))
  p[2:(k+1)] <- prob[2:(k+1)]/(1-prob[1])
  p[1] <- 0
  return(pmf.out(x,p))
}

pztDTS_Pareto <- function(x, alpha, eta, beta){
  k <- max(as.integer(x))
  prob <- dztDTS_Pareto(0:k, alpha, eta, beta)
  c_prob <- cumsum(prob)
  cdf <- stats::stepfun(1:(k), c_prob)
  return(cdf(x))
}

rztDTS_Pareto <- function(n, alpha, eta, beta, c){
  l_plus <- ((gamma(1-alpha) * beta ) / alpha ) * ((1/(beta - alpha)) - beta(alpha+1, beta - alpha))
  N <- actuar::rztpois(n, eta * l_plus)
  Y <- rL_Pareto(sum(N), alpha, eta, beta, c)
  return(sumfun(n, Y, N))
}



logLik_Pareto   = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
    return(100000000000)
  else if(c[3]<= c[1])
    return(100000000000)
  else
  {
    theo = dDTS_Pareto (0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[1:(k+1)] * log(theo[1:(k+1)]))))
  }
}

logLik_Tzt_Pareto = function(c, smpl){
  k <- max(smpl)
  if (c[1] >= 0.95 || c[1] < 0.1)
    return(100000000000)
  else if(c[3]<= c[1])
    return(100000000000)
  else
  {
    theo= dDTS_Pareto(0:k, c[1], c[2], c[3])
    counts = rep(0, k+1)
    for (i in 1:(k+1)){
      counts[i] = sum(smpl == i-1)
    }
    return(-(sum(counts[2:(k+1)] * log(theo[2:(k+1)])) - length(smpl) * log(1 - theo[1])))
  }
}

