% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BayesSurvive.R
\name{BayesSurvive}
\alias{BayesSurvive}
\title{Fit Bayesian Cox Models}
\usage{
BayesSurvive(
  survObj,
  model.type = "Pooled",
  MRF2b = FALSE,
  MRF.G = TRUE,
  g.ini = 0,
  hyperpar = NULL,
  initial = NULL,
  nIter = 1,
  burnin = 0,
  thin = 1,
  output_graph_para = FALSE,
  verbose = TRUE,
  cpp = FALSE
)
}
\arguments{
\item{survObj}{a list containing observed data from \code{n} subjects with
components \code{t}, \code{di}, \code{X}. For graphical learning of the
Markov random field prior, \code{survObj} should be a list of the list with
survival and covariates data. For subgroup models with or without graphical
learning, \code{survObj} should be a list of multiple lists with each
component list representing each subgroup's survival and covariates data}

\item{model.type}{a method option from
\code{c("Pooled", "CoxBVSSL", "Sub-struct")}. To enable graphical learning
for "Pooled" model, please specify \code{list(survObj)} where \code{survObj}
is the list of \code{t}, \code{di} and \code{X}}

\item{MRF2b}{logical value. \code{MRF2b = TRUE} means two different
hyperparameters b in MRF prior (values b01 and b02) and \code{MRF2b = FALSE}
means one hyperparameter b in MRF prior}

\item{MRF.G}{logical value. \code{MRF.G = TRUE} is to fix the MRF graph which
is provided in the argument \code{hyperpar}, and \code{MRF.G = FALSE} is to
use graphical model for learning the MRF graph}

\item{g.ini}{initial values for latent edge inclusion indicators in graph,
should be a value in [0,1]. 0 or 1: set all random edges to 0 or 1; value in
(0,1): rate of indicators randomly set to 1, the remaining indicators are 0}

\item{hyperpar}{a list containing prior parameter values}

\item{initial}{a list containing prior parameters' initial values}

\item{nIter}{the number of iterations of the chain}

\item{burnin}{number of iterations to discard at the start of the chain.
Default is 0}

\item{thin}{thinning MCMC intermediate results to be stored}

\item{output_graph_para}{allow (\code{TRUE}) or suppress (\code{FALSE}) the
output for parameters 'G', 'V', 'C' and 'Sig' in the graphical model
if \code{MRF.G = FALSE}}

\item{verbose}{logical value to display the progress of MCMC}

\item{cpp}{logical, whether to use C++ code for faster computation}
}
\value{
An object of class \code{BayesSurvive} is saved as
\code{obj_BayesSurvive.rda} in the output file, including the following components:
\itemize{
\item input - a list of all input parameters by the user
\item output - a list of the all output estimates:
\itemize{
\item "\code{gamma.p}" - a matrix with MCMC intermediate estimates of the indicator variables of regression coefficients.
\item "\code{beta.p}" - a matrix with MCMC intermediate estimates of the regression coefficients.
\item "\code{h.p}" - a matrix with MCMC intermediate estimates of the increments in the cumulative baseline hazard in each interval.
}
\item call - the matched call.
}
}
\description{
This is the main function to fit a Bayesian Cox model with graph-structured
selection priors for sparse identification of high-dimensional covariates.
}
\examples{

library("BayesSurvive")
set.seed(123)

# Load the example dataset
data("simData", package = "BayesSurvive")

dataset <- list(
  "X" = simData[[1]]$X,
  "t" = simData[[1]]$time,
  "di" = simData[[1]]$status
)

# Initial value: null model without covariates
initial <- list("gamma.ini" = rep(0, ncol(dataset$X)))
# Hyperparameters
hyperparPooled <- list(
  "c0"     = 2, # prior of baseline hazard
  "tau"    = 0.0375, # sd (spike) for coefficient prior
  "cb"     = 20, # sd (spike) for coefficient prior
  "pi.ga"  = 0.02, # prior variable selection probability for standard Cox models
  "a"      = -4, # hyperparameter in MRF prior
  "b"      = 0.1, # hyperparameter in MRF prior
  "G"      = simData$G # hyperparameter in MRF prior
)

\donttest{
# run Bayesian Cox with graph-structured priors
fit <- BayesSurvive(
  survObj = dataset, hyperpar = hyperparPooled,
  initial = initial, nIter = 50
)

# show posterior mean of coefficients and 95\% credible intervals
library("GGally")
plot(fit) +
  coord_flip() +
  theme(axis.text.x = element_text(angle = 90, size = 7))
}

}
