% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bsFisher.R
\name{bsFisher}
\alias{bsFisher}
\alias{bsFisher_setup}
\title{Bayesian Single-Index Regression with B-Spline Link and von Mises-Fisher Prior}
\usage{
bsFisher(
  formula,
  data,
  prior = NULL,
  init = NULL,
  monitors = NULL,
  niter = 10000,
  nburnin = 1000,
  thin = 1,
  nchain = 1,
  setSeed = FALSE
)

bsFisher_setup(
  formula,
  data,
  prior = NULL,
  init = NULL,
  monitors = NULL,
  niter = 10000,
  nburnin = 1000,
  thin = 1,
  nchain = 1,
  setSeed = FALSE
)
}
\arguments{
\item{formula}{an object of class \link{formula}. Interaction term is not allowed for single-index model.}

\item{data}{an data frame.}

\item{prior}{Optional named list of prior settings. Further descriptions are in \strong{Details} section.}

\item{init}{Optional named list of initial values. If the values are not assigned, they are randomly sampled from prior or designated value.
Further descriptions are in \strong{Details} section.}

\item{monitors}{Optional character vector of additional monitor nodes. To check the names of the nodes, fit the \code{model_setup} function and
then inspect the variable names stored in the model object using \link{getVarMonitor}.}

\item{niter}{Integer. Total MCMC iterations (default \code{10000}).}

\item{nburnin}{Integer. Burn-in iterations (default \code{1000}).}

\item{thin}{Integer. Thinning for monitors (default \code{1}).}

\item{nchain}{Integer. Number of MCMC chains (default \code{1}).}

\item{setSeed}{Logical or numeric argument.  Further details are provided in \link[nimble]{runMCMC} \code{setSeed} argument.}
}
\value{
A \code{list} typically containing:
\describe{
\item{\code{coefficients}}{Mean estimates of index vector. Return list of \code{model_setup} does not include it.}
\item{\code{ses}}{Mean standard error of index vector. Return list of \code{model_setup} does not include it.}
\item{\code{residuals}}{Residuals with mean estimates of fitted values. Return list of \code{model_setup} does not include it.}
\item{\code{fitted.values}}{Mean estimates of fitted values. Return list of \code{model_setup} does not include it.}
\item{\code{linear.predictors}}{Mean estimates of single-index values. Return list of \code{model_setup} does not include it.}
\item{\code{gof}}{Goodness-of-fit. Return list of \code{model_setup} function does not include it.}
\item{\code{samples}}{Posterior draws of variables for computing fitted values of the model, including \eqn{\theta}, \eqn{\sigma^2}.
Return list of \code{model_setup} does not include it.}
\item{\code{input}}{List of data used in modeling, formula, input values for prior and initial values, and computation time without compiling.}
\item{\code{defModel}}{Nimble model object.}
\item{\code{defSampler}}{Nimble sampler object.}
\item{\code{modelName}}{Name of the model.}
}
}
\description{
Fits a single-index model \eqn{Y_i \sim \mathcal{N}(f(X_i'\theta), \sigma^2), i = 1,\cdots,n}
where the link \eqn{f(\cdot)} is represented by B-spline and the
index vector \eqn{\theta} has von Mises–Fisher prior.
}
\details{
\strong{Model} The single–index model uses a \eqn{m}-order polynomial spline with \eqn{k} interior knots as follows:
\deqn{f(t) = \sum_{j=1}^{m+k} B_j(t)\,\beta_j} on \eqn{[a, b]} with \eqn{t_i = X_i' \theta, i = 1,\cdots, n}
and \eqn{\|\theta\|_2 = 1}. \eqn{\{\beta_j\}_{j=1}^{m+k}} are spline coefficients and \eqn{a_\theta}, \eqn{b_\theta} are boundary knots where \eqn{a_{\theta} = min(t_i, i = 1, \cdots, n) - \delta},
and \eqn{b_{\theta} = max(t_i, i = 1,\cdots, n) + \delta}.

\strong{Priors}
\itemize{
\item von Mises–Fisher prior on the index \eqn{\theta} with direction and concentration.
\item Conditioned on \eqn{\theta} and \eqn{\sigma^2}, the link coefficients \eqn{\beta = (\beta_1,\ldots,\beta_{m+k})^\top} follow
normal distribution with estimated mean vector \eqn{\hat{\beta}_{\theta} = (X_{\theta}'X_{\theta})^{-1}X_{\theta}'Y} and
covariance \eqn{\sigma^2 (X_{\theta}^\top X_{\theta})^{-1}}, where \eqn{X_{\theta}} is the B-spline basis design matrix.
\item Inverse gamma prior on \eqn{\sigma^2} with shape parameter \eqn{a_\sigma} and rate parameter \eqn{b_\sigma}.
}

\strong{Sampling}
Random walk metropolis algorithm is used for index vector \eqn{\theta}. Given \eqn{\theta}, \eqn{\sigma^2} and \eqn{\beta} are sampled from posterior distribution. Further sampling method is described in Antoniadis et al(2004).

\strong{Prior hyper-parameters}
These are the prior hyper-parameters set in the function. You can define new values for each parameter in \link{prior_param}.
\enumerate{
\item Index vector: von Mises--Fisher prior for the projection vector \eqn{\theta} (\code{index}).
\code{index_direction} is a unit direction vector of the von Mises--Fisher distribution.
By default, the estimated vector from projection pursuit regression is assigned.
\code{index_dispersion} is the positive concentration parameter. By default, \code{150} is assigned.

\item Link function: B-spline basis and coefficient of B-spline setup.
\itemize{
\item \code{basis}: For the basis of B-spline, \code{link_basis_df} is the number of basis functions (default \code{21}),
\code{link_basis_degree} is the spline degree (default \code{2}) and
\code{link_basis_delta} is a small jitter for boundary knots spacing control (default \code{0.001}).
\item \code{beta}: For the coefficient of B-spline, multivariate normal prior is assigned with mean \code{link_beta_mu}, and covariance \code{link_beta_cov}.
By default, \eqn{\mathcal{N}_p\!\big(0, \mathrm{I}_p\big)} is assigned.
}

\item Error variance (\code{sigma2}): An Inverse gamma prior is assigned to \eqn{\sigma^2}
where \code{sigma2_shape} is shape parameter and \code{sigma2_rate} is rate parameter of inverse gamma distribution.
(default \code{sigma2_shape = 0.001, sigma2_rate = 100})
}

\strong{Initial values}
These are the initial values set in the function. You can define new values for each initial value in \link{init_param}.
\enumerate{
\item Index vector: Initial unit index vector \eqn{\theta}. By default, the vector is randomly sampled from the von Mises--Fisher prior.
\item Link function: Initial spline coefficients (\code{link_beta}). By default,
\eqn{\big(X_{\theta}^\top X_{\theta} + \rho\, \mathrm{I}\big)^{-1} X_{\theta}^\top Y} is computed,
where \eqn{X_{\theta}} is the B-spline basis design matrix.

\item Error variance (\code{sigma2}): Initial scalar error variance (default \code{0.01}).
}
}
\examples{
\donttest{
set.seed(123)
n <- 200; d <- 4
theta <- c(2, 1, 1, 1); theta <- theta / sqrt(sum(theta^2))
f <- function(u) u^2 * exp(u)
sigma <- 0.5
X <- matrix(runif(n * d, -1, 1), nrow = n)
index_vals <- as.vector(X \%*\% theta)
y <- f(index_vals) + rnorm(n, 0, sigma)
simdata <- data.frame(x = X, y = y)
colnames(simdata) <- c(paste0("X", 1:4), "y")

# One tool version
fit1 <- bsFisher(y ~ ., data = simdata,
                 niter = 5000, nburnin = 1000, nchain = 1)

# Split version
models <- bsFisher_setup(y ~ ., data = simdata)
Ccompile <- compileModelAndMCMC(models)
nimSampler <- get_sampler(Ccompile)
initList <- getInit(models)
mcmc.out <- runMCMC(nimSampler, niter = 5000, nburnin = 1000, thin = 1,
                   nchains = 1, setSeed = TRUE, inits = initList,
                   summary = TRUE, samplesAsCodaMCMC = TRUE)
fit2 <- as_bsim(models, mcmc.out)
summary(fit2)


}

}
\references{
Antoniadis, A., Grégoire, G., & McKeague, I. W. (2004).
Bayesian estimation in single-index models. \emph{Statistica Sinica}, 1147-1164.

Hornik, K., & Grün, B. (2014). movMF: an R package for fitting mixtures of von Mises-Fisher distributions.
\emph{Journal of Statistical Software}, 58, 1-31.
}
