#' Generate a Directed Acyclic Graph (DAG) randomly
#'
#' This function randomly generates a Directed Acyclic Graph (DAG) with \code{q} nodes and probability of edge inclusion \code{w}.
#'
#'
#' The \eqn{0-1} adjacency matrix of the DAG is generated by drawing each element in the lower triangular part in \eqn{{0,1}} with probability \eqn{{1-w, w}}.
#' Accordingly, the DAG has lower-triangular adjacency matrix and nodes are numerically labeled according to a topological ordering implying \eqn{u > v} whenever \eqn{u -> v}.
#'
#' @param q number of nodes
#' @param w probability of edge inclusion in \eqn{[0,1]}
#'
#' @return DAG \eqn{(q,q)} adjacency matrix of the generated DAG
#' @export
#'
#' @examples # Randomly generate a DAG on q = 8 nodes with probability of edge inclusion w = 0.2
#' q = 8
#' w = 0.2
#' set.seed(123)
#' rDAG(q = q, w = w)
#'
rDAG = function(q, w){
  DAG = matrix(0, q, q); colnames(DAG) = rownames(DAG) = 1:q
  DAG[lower.tri(DAG)] = stats::rbinom(n = q*(q-1)/2, size = 1, prob = w)
  return(DAG)
}
