% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/keys.R
\docType{class}
\name{keys}
\alias{keys}
\title{Encryption keys in Key Vault}
\description{
This class represents the collection of encryption keys stored in a vault. It provides methods for managing keys, including creating, importing and deleting keys, and doing backups and restores. For operations with a specific key, see \link{key}.
}
\section{Methods}{

This class provides the following methods:

\if{html}{\out{<div class="sourceCode">}}\preformatted{create(name, type=c("RSA", "EC"), hardware=FALSE,
       ec_curve=NULL, rsa_key_size=NULL, key_ops=NULL,
       attributes=vault_object_attrs(), ...)
import(name, key, hardware=FALSE,
       attributes=vault_object_attrs(), ...)
get(name)
delete(name, confirm=TRUE)
list(include_managed=FALSE)
backup(name)
restore(backup)
}\if{html}{\out{</div>}}
}

\section{Arguments}{

\itemize{
\item \code{name}: The name of the key.
\item \code{type}: For \code{create}, the type of key to create: RSA or elliptic curve (EC). Note that for keys backing a certificate, only RSA is allowed.
\item \code{hardware}: For \code{create}, Whether to use a hardware key or software key. The former requires a premium key vault.
\item \code{ec_curve}: For an EC key, the type of elliptic curve.
\item \code{rsa_key_size}: For an RSA key, the key size, either 2048, 3072 or 4096.
\item \code{key_ops}: A character vector of operations that the key supports. The possible operations are "encrypt", "decrypt", "sign", "verify", "wrapkey" and "unwrapkey". See \link{key} for more information.
\item \code{attributes}: Optional attributes for the key, such as the expiry date and activation date. A convenient way to provide this is via the \link{vault_object_attrs} helper function.
\item \code{key}: For \code{import}, the key to import. This can be the name of a PEM file, a JSON web key (JWK) string, or a key object generated by the openssl package. See the examples below.
\item \code{hardware}: For \code{import}, whether to import this key as a hardware key (HSM). Only supported for a premium key vault.
\item \code{...}: For \code{create} and \code{import}, other named arguments which will be treated as tags.
\item \code{include_managed}: For \code{list}, whether to include keys that were created by Key Vault to support a managed certificate.
\item \code{backup}: For \code{restore}, a string representing the backup blob for a key.
}
}

\section{Value}{

For \code{get}, \code{create} and \code{import}, an object of class \code{stored_key}, representing the key itself. This has methods for carrying out the operations given by the \code{key_ops} argument.

For \code{list}, a vector of key names.

For \code{backup}, a string representing the backup blob for a key. If the key has multiple versions, the blob will contain all versions.
}

\examples{
\dontrun{

vault <- key_vault("mykeyvault")

vault$keys$create("mynewkey")
vault$keys$create("myRSAkey", type="RSA", rsa_key_size=4096)
vault$keys$create("myECkey", type="EC", ec_curve="P-384")

vault$keys$list()
vault$keys$get("mynewkey")

# specifying an expiry date
today <- Sys.date()
vault$keys$create("mynewkey", attributes=vault_object_attrs(expiry_date=today+365))

# setting management tags
vault$keys$create("mynewkey", tag1="a value", othertag="another value")

# importing a key from a PEM file
vault$keys$import("importedkey1", "myprivatekey.pem")

# importing a key generated by OpenSSL
vault$keys$import("importedkey2", openssl::rsa_keygen())

# importing a JWK (which is a JSON string)
key <- openssl::read_key("myprivatekey.pem")
jwk <- jose::write_jwk(key)
vault$keys$import("importedkey3", jwk)

# backup and restore a key
bak <- vault$keys$backup("mynewkey")
vault$keys$delete("mynewkey", confirm=FALSE)
vault$keys$restore(bak)

}
}
\seealso{
\link{key}, \link{vault_object_attrs}

\href{https://learn.microsoft.com/en-us/azure/key-vault/}{Azure Key Vault documentation},
\href{https://learn.microsoft.com/en-us/rest/api/keyvault}{Azure Key Vault API reference}
}
