/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.UserPrincipal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.Build;
import org.elasticsearch.ElasticsearchCorruptionException;
import org.elasticsearch.ElasticsearchTimeoutException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.cli.Terminal;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.http.client.HttpDownloadHelper;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.env.Environment;
import org.elasticsearch.plugins.PluginInfo;
import org.elasticsearch.plugins.PluginSecurity;
import org.elasticsearch.plugins.PluginsService;

public class PluginManager {
    public static final String PROPERTY_SUPPORT_STAGING_URLS = "es.plugins.staging";
    private static final ImmutableSet<String> BLACKLIST = ((ImmutableSet.Builder)ImmutableSet.builder().add(new String[]{"elasticsearch", "elasticsearch.bat", "elasticsearch.in.sh", "plugin", "plugin.bat", "service.bat"})).build();
    static final Set<String> MODULES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("lang-expression", "lang-groovy")));
    static final ImmutableSet<String> OFFICIAL_PLUGINS = ((ImmutableSet.Builder)ImmutableSet.builder().add(new String[]{"analysis-icu", "analysis-kuromoji", "analysis-phonetic", "analysis-smartcn", "analysis-stempel", "cloud-aws", "cloud-azure", "cloud-gce", "delete-by-query", "discovery-multicast", "lang-javascript", "lang-python", "mapper-attachments", "mapper-murmur3", "mapper-size"})).build();
    private final Environment environment;
    private URL url;
    private OutputMode outputMode;
    private TimeValue timeout;

    public PluginManager(Environment environment, URL url, OutputMode outputMode, TimeValue timeout) {
        this.environment = environment;
        this.url = url;
        this.outputMode = outputMode;
        this.timeout = timeout;
    }

    public void downloadAndExtract(String name, Terminal terminal, boolean batch) throws IOException {
        PluginHandle pluginHandle;
        if (name == null && this.url == null) {
            throw new IllegalArgumentException("plugin name or url must be supplied with install.");
        }
        if (!Files.exists(this.environment.pluginsFile(), new LinkOption[0])) {
            terminal.println("Plugins directory [%s] does not exist. Creating...", this.environment.pluginsFile());
            Files.createDirectory(this.environment.pluginsFile(), new FileAttribute[0]);
        }
        if (!Environment.isWritable(this.environment.pluginsFile())) {
            throw new IOException("plugin directory " + this.environment.pluginsFile() + " is read only");
        }
        if (name != null) {
            pluginHandle = PluginHandle.parse(name);
            PluginManager.checkForForbiddenName(pluginHandle.name);
        } else {
            pluginHandle = new PluginHandle("temp_name" + new Random().nextInt(), null, null);
        }
        Path pluginFile = this.download(pluginHandle, terminal);
        this.extract(pluginHandle, terminal, pluginFile, batch);
    }

    private Path download(PluginHandle pluginHandle, Terminal terminal) throws IOException {
        Path pluginFile = pluginHandle.newDistroFile(this.environment);
        HttpDownloadHelper downloadHelper = new HttpDownloadHelper();
        boolean downloaded = false;
        boolean verified = false;
        HttpDownloadHelper.DownloadProgress progress = this.outputMode == OutputMode.SILENT ? new HttpDownloadHelper.NullProgress() : new HttpDownloadHelper.VerboseProgress(terminal.writer());
        if (this.url != null) {
            boolean isAuthInfoSet;
            URL pluginUrl = this.url;
            boolean isSecureProcotol = "https".equalsIgnoreCase(pluginUrl.getProtocol());
            boolean bl = isAuthInfoSet = !Strings.isNullOrEmpty(pluginUrl.getUserInfo());
            if (isAuthInfoSet && !isSecureProcotol) {
                throw new IOException("Basic auth is only supported for HTTPS!");
            }
            terminal.println("Trying %s ...", pluginUrl.toExternalForm());
            try {
                downloadHelper.download(pluginUrl, pluginFile, progress, this.timeout);
                downloaded = true;
                terminal.println("Verifying %s checksums if available ...", pluginUrl.toExternalForm());
                Tuple<URL, Path> sha1Info = pluginHandle.newChecksumUrlAndFile(this.environment, pluginUrl, "sha1");
                verified = downloadHelper.downloadAndVerifyChecksum(sha1Info.v1(), pluginFile, sha1Info.v2(), progress, this.timeout, HttpDownloadHelper.SHA1_CHECKSUM);
                Tuple<URL, Path> md5Info = pluginHandle.newChecksumUrlAndFile(this.environment, pluginUrl, "md5");
                verified = verified || downloadHelper.downloadAndVerifyChecksum(md5Info.v1(), pluginFile, md5Info.v2(), progress, this.timeout, HttpDownloadHelper.MD5_CHECKSUM);
            }
            catch (ElasticsearchCorruptionException | ElasticsearchTimeoutException e) {
                throw e;
            }
            catch (Exception e) {
                terminal.println("Failed: %s", ExceptionsHelper.detailedMessage(e));
            }
        } else if (PluginHandle.isOfficialPlugin(pluginHandle.name, pluginHandle.user, pluginHandle.version)) {
            PluginManager.checkForOfficialPlugins(pluginHandle.name);
        }
        if (!downloaded && this.url == null) {
            for (URL url : pluginHandle.urls()) {
                terminal.println("Trying %s ...", url.toExternalForm());
                try {
                    downloadHelper.download(url, pluginFile, progress, this.timeout);
                    downloaded = true;
                    terminal.println("Verifying %s checksums if available ...", url.toExternalForm());
                    Tuple<URL, Path> sha1Info = pluginHandle.newChecksumUrlAndFile(this.environment, url, "sha1");
                    verified = downloadHelper.downloadAndVerifyChecksum(sha1Info.v1(), pluginFile, sha1Info.v2(), progress, this.timeout, HttpDownloadHelper.SHA1_CHECKSUM);
                    Tuple<URL, Path> md5Info = pluginHandle.newChecksumUrlAndFile(this.environment, url, "md5");
                    verified = verified || downloadHelper.downloadAndVerifyChecksum(md5Info.v1(), pluginFile, md5Info.v2(), progress, this.timeout, HttpDownloadHelper.MD5_CHECKSUM);
                    break;
                }
                catch (ElasticsearchCorruptionException | ElasticsearchTimeoutException e) {
                    throw e;
                }
                catch (Exception e) {
                    terminal.println(Terminal.Verbosity.VERBOSE, "Failed: %s", ExceptionsHelper.detailedMessage(e));
                }
            }
        }
        if (!downloaded) {
            IOUtils.deleteFilesIgnoringExceptions(pluginFile);
            throw new IOException("failed to download out of all possible locations..., use --verbose to get detailed information");
        }
        if (!verified) {
            terminal.println("NOTE: Unable to verify checksum for downloaded plugin (unable to find .sha1 or .md5 file to verify)", new Object[0]);
        }
        return pluginFile;
    }

    private void extract(PluginHandle pluginHandle, Terminal terminal, Path pluginFile, boolean batch) throws IOException {
        Path policy;
        Path tmp = Files.createTempDirectory(this.environment.tmpFile(), null, new FileAttribute[0]);
        Path root = tmp.resolve(pluginHandle.name);
        this.unzipPlugin(pluginFile, root);
        root = this.findPluginRoot(root);
        PluginInfo info = PluginInfo.readFromProperties(root);
        terminal.println(Terminal.Verbosity.VERBOSE, "%s", info);
        if (MODULES.contains(info.getName())) {
            throw new IOException("plugin '" + info.getName() + "' cannot be installed like this, it is a system module");
        }
        pluginHandle = new PluginHandle(info.getName(), pluginHandle.version, pluginHandle.user);
        Path extractLocation = pluginHandle.extractedDir(this.environment);
        if (Files.exists(extractLocation, new LinkOption[0])) {
            throw new IOException("plugin directory " + extractLocation.toAbsolutePath() + " already exists. To update the plugin, uninstall it first using 'remove " + pluginHandle.name + "' command");
        }
        if (info.isJvm()) {
            this.jarHellCheck(root, info.isIsolated());
        }
        if (Files.exists(policy = root.resolve("plugin-security.policy"), new LinkOption[0])) {
            PluginSecurity.readPolicy(policy, terminal, this.environment, batch);
        }
        FileSystemUtils.copyDirectoryRecursively(root, extractLocation);
        terminal.println("Installed %s into %s", pluginHandle.name, extractLocation.toAbsolutePath());
        PluginManager.tryToDeletePath(terminal, tmp, pluginFile);
        Path sourcePluginBinDirectory = extractLocation.resolve("bin");
        Path destPluginBinDirectory = pluginHandle.binDir(this.environment);
        boolean needToCopyBinDirectory = Files.exists(sourcePluginBinDirectory, new LinkOption[0]);
        if (needToCopyBinDirectory) {
            if (Files.exists(destPluginBinDirectory, new LinkOption[0]) && !Files.isDirectory(destPluginBinDirectory, new LinkOption[0])) {
                PluginManager.tryToDeletePath(terminal, extractLocation);
                throw new IOException("plugin bin directory " + destPluginBinDirectory + " is not a directory");
            }
            try {
                this.copyBinDirectory(sourcePluginBinDirectory, destPluginBinDirectory, pluginHandle.name, terminal);
            }
            catch (IOException e) {
                terminal.printError("Error copying bin directory [%s] to [%s], cleaning up, reason: %s", sourcePluginBinDirectory, destPluginBinDirectory, ExceptionsHelper.detailedMessage(e));
                PluginManager.tryToDeletePath(terminal, extractLocation, pluginHandle.binDir(this.environment));
                throw e;
            }
        }
        Path sourceConfigDirectory = extractLocation.resolve("config");
        Path destConfigDirectory = pluginHandle.configDir(this.environment);
        boolean needToCopyConfigDirectory = Files.exists(sourceConfigDirectory, new LinkOption[0]);
        if (needToCopyConfigDirectory) {
            if (Files.exists(destConfigDirectory, new LinkOption[0]) && !Files.isDirectory(destConfigDirectory, new LinkOption[0])) {
                PluginManager.tryToDeletePath(terminal, extractLocation, destPluginBinDirectory);
                throw new IOException("plugin config directory " + destConfigDirectory + " is not a directory");
            }
            try {
                terminal.println(Terminal.Verbosity.VERBOSE, "Found config, moving to %s", destConfigDirectory.toAbsolutePath());
                FileSystemUtils.moveFilesWithoutOverwriting(sourceConfigDirectory, destConfigDirectory, ".new");
                if (Environment.getFileStore(destConfigDirectory).supportsFileAttributeView(PosixFileAttributeView.class)) {
                    final PosixFileAttributes parentDirAttributes = Files.getFileAttributeView(destConfigDirectory.getParent(), PosixFileAttributeView.class, new LinkOption[0]).readAttributes();
                    final HashSet<PosixFilePermission> baseFilePermissions = new HashSet<PosixFilePermission>();
                    block7: for (PosixFilePermission posixFilePermission : parentDirAttributes.permissions()) {
                        switch (posixFilePermission) {
                            case OWNER_EXECUTE: 
                            case GROUP_EXECUTE: 
                            case OTHERS_EXECUTE: {
                                continue block7;
                            }
                        }
                        baseFilePermissions.add(posixFilePermission);
                    }
                    Files.walkFileTree(destConfigDirectory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                        @Override
                        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                            if (attrs.isRegularFile()) {
                                HashSet<PosixFilePermission> newFilePermissions = new HashSet<PosixFilePermission>(baseFilePermissions);
                                Set<PosixFilePermission> currentFilePermissions = Files.getPosixFilePermissions(file, new LinkOption[0]);
                                for (PosixFilePermission posixFilePermission : currentFilePermissions) {
                                    switch (posixFilePermission) {
                                        case OWNER_EXECUTE: 
                                        case GROUP_EXECUTE: 
                                        case OTHERS_EXECUTE: {
                                            newFilePermissions.add(posixFilePermission);
                                        }
                                    }
                                }
                                PluginManager.setPosixFileAttributes(file, parentDirAttributes.owner(), parentDirAttributes.group(), newFilePermissions);
                            }
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                            PluginManager.setPosixFileAttributes(dir, parentDirAttributes.owner(), parentDirAttributes.group(), parentDirAttributes.permissions());
                            return FileVisitResult.CONTINUE;
                        }
                    });
                } else {
                    terminal.println(Terminal.Verbosity.VERBOSE, "Skipping posix permissions - filestore doesn't support posix permission", new Object[0]);
                }
                terminal.println(Terminal.Verbosity.VERBOSE, "Installed %s into %s", pluginHandle.name, destConfigDirectory.toAbsolutePath());
            }
            catch (IOException e) {
                terminal.printError("Error copying config directory [%s] to [%s], cleaning up, reason: %s", sourceConfigDirectory, destConfigDirectory, ExceptionsHelper.detailedMessage(e));
                PluginManager.tryToDeletePath(terminal, extractLocation, destPluginBinDirectory, destConfigDirectory);
                throw e;
            }
        }
    }

    private static void setPosixFileAttributes(Path path, UserPrincipal owner, GroupPrincipal group, Set<PosixFilePermission> permissions) throws IOException {
        PosixFileAttributeView fileAttributeView = Files.getFileAttributeView(path, PosixFileAttributeView.class, new LinkOption[0]);
        fileAttributeView.setOwner(owner);
        fileAttributeView.setGroup(group);
        fileAttributeView.setPermissions(permissions);
    }

    static void tryToDeletePath(Terminal terminal, Path ... paths) {
        for (Path path : paths) {
            try {
                IOUtils.rm(path);
            }
            catch (IOException e) {
                terminal.printError(e);
            }
        }
    }

    private void copyBinDirectory(Path sourcePluginBinDirectory, Path destPluginBinDirectory, String pluginName, Terminal terminal) throws IOException {
        boolean canCopyFromSource;
        boolean bl = canCopyFromSource = Files.exists(sourcePluginBinDirectory, new LinkOption[0]) && Files.isReadable(sourcePluginBinDirectory) && Files.isDirectory(sourcePluginBinDirectory, new LinkOption[0]);
        if (canCopyFromSource) {
            terminal.println(Terminal.Verbosity.VERBOSE, "Found bin, moving to %s", destPluginBinDirectory.toAbsolutePath());
            if (Files.exists(destPluginBinDirectory, new LinkOption[0])) {
                IOUtils.rm(destPluginBinDirectory);
            }
            try {
                Files.createDirectories(destPluginBinDirectory.getParent(), new FileAttribute[0]);
                FileSystemUtils.move(sourcePluginBinDirectory, destPluginBinDirectory);
            }
            catch (IOException e) {
                throw new IOException("Could not move [" + sourcePluginBinDirectory + "] to [" + destPluginBinDirectory + "]", e);
            }
            if (Environment.getFileStore(destPluginBinDirectory).supportsFileAttributeView(PosixFileAttributeView.class)) {
                final PosixFileAttributes parentDirAttributes = Files.getFileAttributeView(destPluginBinDirectory.getParent(), PosixFileAttributeView.class, new LinkOption[0]).readAttributes();
                final HashSet<PosixFilePermission> filePermissions = new HashSet<PosixFilePermission>();
                block5: for (PosixFilePermission posixFilePermission : parentDirAttributes.permissions()) {
                    switch (posixFilePermission) {
                        case OWNER_EXECUTE: 
                        case GROUP_EXECUTE: 
                        case OTHERS_EXECUTE: {
                            continue block5;
                        }
                    }
                    filePermissions.add(posixFilePermission);
                }
                filePermissions.add(PosixFilePermission.OWNER_EXECUTE);
                filePermissions.add(PosixFilePermission.GROUP_EXECUTE);
                filePermissions.add(PosixFilePermission.OTHERS_EXECUTE);
                Files.walkFileTree(destPluginBinDirectory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        if (attrs.isRegularFile()) {
                            PluginManager.setPosixFileAttributes(file, parentDirAttributes.owner(), parentDirAttributes.group(), filePermissions);
                        }
                        return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                        PluginManager.setPosixFileAttributes(dir, parentDirAttributes.owner(), parentDirAttributes.group(), parentDirAttributes.permissions());
                        return FileVisitResult.CONTINUE;
                    }
                });
            } else {
                terminal.println(Terminal.Verbosity.VERBOSE, "Skipping posix permissions - filestore doesn't support posix permission", new Object[0]);
            }
            terminal.println(Terminal.Verbosity.VERBOSE, "Installed %s into %s", pluginName, destPluginBinDirectory.toAbsolutePath());
        }
    }

    private Path findPluginRoot(Path dir) throws IOException {
        Path subdir;
        if (Files.exists(dir.resolve("plugin-descriptor.properties"), new LinkOption[0])) {
            return dir;
        }
        Path[] topLevelFiles = FileSystemUtils.files(dir);
        if (topLevelFiles.length == 1 && Files.isDirectory(topLevelFiles[0], new LinkOption[0]) && Files.exists((subdir = topLevelFiles[0]).resolve("plugin-descriptor.properties"), new LinkOption[0])) {
            return subdir;
        }
        throw new RuntimeException("Could not find plugin descriptor 'plugin-descriptor.properties' in plugin zip");
    }

    private void jarHellCheck(Path candidate, boolean isolated) throws IOException {
        Path[] pluginJars;
        ArrayList<URL> jars = new ArrayList<URL>();
        jars.addAll(Arrays.asList(JarHell.parseClassPath()));
        List<PluginsService.Bundle> bundles = PluginsService.getPluginBundles(this.environment.pluginsFile());
        if (!isolated) {
            jars.addAll(bundles.get((int)0).urls);
        }
        for (Path jar : pluginJars = FileSystemUtils.files(candidate, "*.jar")) {
            jars.add(jar.toUri().toURL());
        }
        try {
            JarHell.checkJarHell(jars.toArray(new URL[jars.size()]));
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    private void unzipPlugin(Path zip, Path target) throws IOException {
        Files.createDirectories(target, new FileAttribute[0]);
        try (ZipInputStream zipInput = new ZipInputStream(Files.newInputStream(zip, new OpenOption[0]));){
            ZipEntry entry;
            byte[] buffer = new byte[8192];
            while ((entry = zipInput.getNextEntry()) != null) {
                Path targetFile = target.resolve(entry.getName());
                Files.createDirectories(targetFile.getParent(), new FileAttribute[0]);
                if (!entry.isDirectory()) {
                    try (OutputStream out = Files.newOutputStream(targetFile, new OpenOption[0]);){
                        int len;
                        while ((len = zipInput.read(buffer)) >= 0) {
                            out.write(buffer, 0, len);
                        }
                    }
                }
                zipInput.closeEntry();
            }
        }
    }

    public void removePlugin(String name, Terminal terminal) throws IOException {
        Path binLocation;
        if (name == null) {
            throw new IllegalArgumentException("plugin name must be supplied with remove [name].");
        }
        PluginHandle pluginHandle = PluginHandle.parse(name);
        boolean removed = false;
        PluginManager.checkForForbiddenName(pluginHandle.name);
        Path pluginToDelete = pluginHandle.extractedDir(this.environment);
        if (Files.exists(pluginToDelete, new LinkOption[0])) {
            terminal.println(Terminal.Verbosity.VERBOSE, "Removing: %s", pluginToDelete);
            try {
                IOUtils.rm(pluginToDelete);
            }
            catch (IOException ex) {
                throw new IOException("Unable to remove " + pluginHandle.name + ". Check file permissions on " + pluginToDelete.toString(), ex);
            }
            removed = true;
        }
        if (Files.exists(binLocation = pluginHandle.binDir(this.environment), new LinkOption[0])) {
            terminal.println(Terminal.Verbosity.VERBOSE, "Removing: %s", binLocation);
            try {
                IOUtils.rm(binLocation);
            }
            catch (IOException ex) {
                throw new IOException("Unable to remove " + pluginHandle.name + ". Check file permissions on " + binLocation.toString(), ex);
            }
            removed = true;
        }
        if (removed) {
            terminal.println("Removed %s", name);
        } else {
            terminal.println("Plugin %s not found. Run \"plugin list\" to get list of installed plugins.", name);
        }
    }

    static void checkForForbiddenName(String name) {
        if (!Strings.hasLength(name) || BLACKLIST.contains(name.toLowerCase(Locale.ROOT))) {
            throw new IllegalArgumentException("Illegal plugin name: " + name);
        }
    }

    protected static void checkForOfficialPlugins(String name) {
        if (!OFFICIAL_PLUGINS.contains(name)) {
            throw new IllegalArgumentException(name + " is not an official plugin so you should install it using elasticsearch/" + name + "/latest naming form.");
        }
    }

    public Path[] getListInstalledPlugins() throws IOException {
        if (!Files.exists(this.environment.pluginsFile(), new LinkOption[0])) {
            return new Path[0];
        }
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.environment.pluginsFile());){
            Path[] pathArray = Iterators.toArray(stream.iterator(), Path.class);
            return pathArray;
        }
    }

    public void listInstalledPlugins(Terminal terminal) throws IOException {
        Path[] plugins = this.getListInstalledPlugins();
        terminal.println("Installed plugins in %s:", this.environment.pluginsFile().toAbsolutePath());
        if (plugins == null || plugins.length == 0) {
            terminal.println("    - No plugin detected", new Object[0]);
        } else {
            for (Path plugin : plugins) {
                terminal.println("    - " + plugin.getFileName(), new Object[0]);
            }
        }
    }

    static class PluginHandle {
        final String version;
        final String user;
        final String name;

        PluginHandle(String name, String version, String user) {
            this.version = version;
            this.user = user;
            this.name = name;
        }

        List<URL> urls() {
            ArrayList<URL> urls = new ArrayList<URL>();
            if (this.version != null) {
                if (this.user == null) {
                    if (!Strings.isNullOrEmpty(System.getProperty(PluginManager.PROPERTY_SUPPORT_STAGING_URLS))) {
                        PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://download.elastic.co/elasticsearch/staging/%s-%s/org/elasticsearch/plugin/%s/%s/%s-%s.zip", this.version, Build.CURRENT.hashShort(), this.name, this.version, this.name, this.version));
                    }
                    PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://download.elastic.co/elasticsearch/release/org/elasticsearch/plugin/%s/%s/%s-%s.zip", this.name, this.version, this.name, this.version));
                } else {
                    PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://download.elastic.co/%1$s/%2$s/%2$s-%3$s.zip", this.user, this.name, this.version));
                    PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://search.maven.org/remotecontent?filepath=%1$s/%2$s/%3$s/%2$s-%3$s.zip", this.user.replace('.', '/'), this.name, this.version));
                    PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://oss.sonatype.org/service/local/repositories/releases/content/%1$s/%2$s/%3$s/%2$s-%3$s.zip", this.user.replace('.', '/'), this.name, this.version));
                    PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://github.com/%1$s/%2$s/archive/%3$s.zip", this.user, this.name, this.version));
                }
            }
            if (this.user != null) {
                PluginHandle.addUrl(urls, String.format(Locale.ROOT, "https://github.com/%1$s/%2$s/archive/master.zip", this.user, this.name));
            }
            return urls;
        }

        private static void addUrl(List<URL> urls, String url) {
            try {
                urls.add(new URL(url));
            }
            catch (MalformedURLException malformedURLException) {
                // empty catch block
            }
        }

        Path newDistroFile(Environment env) throws IOException {
            return Files.createTempFile(env.tmpFile(), this.name, ".zip", new FileAttribute[0]);
        }

        Tuple<URL, Path> newChecksumUrlAndFile(Environment env, URL originalUrl, String suffix) throws IOException {
            URL newUrl = new URL(originalUrl.toString() + "." + suffix);
            return new Tuple<URL, Path>(newUrl, Files.createTempFile(env.tmpFile(), this.name, ".zip." + suffix, new FileAttribute[0]));
        }

        Path extractedDir(Environment env) {
            return env.pluginsFile().resolve(this.name);
        }

        Path binDir(Environment env) {
            return env.binFile().resolve(this.name);
        }

        Path configDir(Environment env) {
            return env.configFile().resolve(this.name);
        }

        static PluginHandle parse(String name) {
            String[] elements = name.split("/");
            String repo = elements[0];
            String user = null;
            String version = null;
            if (elements.length > 1) {
                user = elements[0];
                repo = elements[1];
                if (elements.length > 2) {
                    version = elements[2];
                }
            }
            if (PluginHandle.isOfficialPlugin(repo, user, version)) {
                return new PluginHandle(repo, Version.CURRENT.number(), null);
            }
            return new PluginHandle(repo, version, user);
        }

        static boolean isOfficialPlugin(String repo, String user, String version) {
            return version == null && user == null && !Strings.isNullOrEmpty(repo);
        }
    }

    public static enum OutputMode {
        DEFAULT,
        SILENT,
        VERBOSE;

    }
}

