/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.server.storage;

import io.prometheus.client.Counter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.uniffle.common.RemoteStorageInfo;
import org.apache.uniffle.common.UnionKey;
import org.apache.uniffle.common.config.RssConf;
import org.apache.uniffle.common.exception.RssException;
import org.apache.uniffle.common.storage.StorageInfo;
import org.apache.uniffle.common.storage.StorageMedia;
import org.apache.uniffle.common.storage.StorageStatus;
import org.apache.uniffle.common.util.RssUtils;
import org.apache.uniffle.common.util.ThreadUtils;
import org.apache.uniffle.guava.annotations.VisibleForTesting;
import org.apache.uniffle.guava.collect.Lists;
import org.apache.uniffle.guava.collect.Maps;
import org.apache.uniffle.server.Checker;
import org.apache.uniffle.server.LocalStorageChecker;
import org.apache.uniffle.server.ShuffleDataFlushEvent;
import org.apache.uniffle.server.ShuffleDataReadEvent;
import org.apache.uniffle.server.ShuffleServerConf;
import org.apache.uniffle.server.ShuffleServerMetrics;
import org.apache.uniffle.server.event.AppPurgeEvent;
import org.apache.uniffle.server.event.PurgeEvent;
import org.apache.uniffle.server.event.ShufflePurgeEvent;
import org.apache.uniffle.server.storage.SingleStorageManager;
import org.apache.uniffle.storage.common.LocalStorage;
import org.apache.uniffle.storage.common.Storage;
import org.apache.uniffle.storage.common.StorageMediaProvider;
import org.apache.uniffle.storage.factory.ShuffleHandlerFactory;
import org.apache.uniffle.storage.handler.api.ShuffleDeleteHandler;
import org.apache.uniffle.storage.request.CreateShuffleDeleteHandlerRequest;
import org.apache.uniffle.storage.util.ShuffleStorageUtils;
import org.apache.uniffle.storage.util.StorageType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalStorageManager
extends SingleStorageManager {
    private static final Logger LOG = LoggerFactory.getLogger(LocalStorageManager.class);
    private static final String UNKNOWN_USER_NAME = "unknown";
    private final List<LocalStorage> localStorages;
    private final List<String> storageBasePaths;
    private final LocalStorageChecker checker;
    private final ConcurrentSkipListMap<String, LocalStorage> sortedPartitionsOfStorageMap;
    private final List<StorageMediaProvider> typeProviders = Lists.newArrayList();

    @VisibleForTesting
    LocalStorageManager(ShuffleServerConf conf) {
        super(conf);
        double lowWaterMarkOfWrite;
        this.storageBasePaths = RssUtils.getConfiguredLocalDirs((RssConf)conf);
        if (CollectionUtils.isEmpty(this.storageBasePaths)) {
            throw new IllegalArgumentException("Base path dirs must not be empty");
        }
        this.sortedPartitionsOfStorageMap = new ConcurrentSkipListMap();
        long capacity = conf.getSizeAsBytes(ShuffleServerConf.DISK_CAPACITY);
        double ratio = conf.getDouble(ShuffleServerConf.DISK_CAPACITY_RATIO);
        double highWaterMarkOfWrite = (Double)conf.get(ShuffleServerConf.HIGH_WATER_MARK_OF_WRITE);
        if (highWaterMarkOfWrite < (lowWaterMarkOfWrite = ((Double)conf.get(ShuffleServerConf.LOW_WATER_MARK_OF_WRITE)).doubleValue())) {
            throw new IllegalArgumentException("highWaterMarkOfWrite must be larger than lowWaterMarkOfWrite");
        }
        CountDownLatch countDownLatch = new CountDownLatch(this.storageBasePaths.size());
        AtomicInteger successCount = new AtomicInteger();
        ServiceLoader<StorageMediaProvider> loader = ServiceLoader.load(StorageMediaProvider.class);
        for (StorageMediaProvider provider : loader) {
            provider.init((RssConf)conf);
            this.typeProviders.add(provider);
        }
        ExecutorService executorService = ThreadUtils.getDaemonCachedThreadPool((String)"LocalStorage-check");
        LocalStorage[] localStorageArray = new LocalStorage[this.storageBasePaths.size()];
        boolean isDiskCapacityWatermarkCheckEnabled = (Boolean)conf.get(ShuffleServerConf.DISK_CAPACITY_WATERMARK_CHECK_ENABLED);
        for (int i = 0; i < this.storageBasePaths.size(); ++i) {
            int idx = i;
            String storagePath = this.storageBasePaths.get(i);
            executorService.submit(() -> {
                try {
                    StorageMedia storageType = this.getStorageTypeForBasePath(storagePath);
                    LocalStorage.Builder builder = LocalStorage.newBuilder().basePath(storagePath).capacity(capacity).ratio(ratio).lowWaterMarkOfWrite(lowWaterMarkOfWrite).highWaterMarkOfWrite(highWaterMarkOfWrite).localStorageMedia(storageType);
                    if (isDiskCapacityWatermarkCheckEnabled) {
                        builder.enableDiskCapacityWatermarkCheck();
                    }
                    localStorageArray[idx] = builder.build();
                    successCount.incrementAndGet();
                }
                catch (Exception e) {
                    LOG.error("LocalStorage init failed!", (Throwable)e);
                }
                finally {
                    countDownLatch.countDown();
                }
            });
        }
        try {
            countDownLatch.await();
        }
        catch (InterruptedException e) {
            LOG.error("Failed to wait initializing local storage.", (Throwable)e);
        }
        executorService.shutdown();
        int failedCount = this.storageBasePaths.size() - successCount.get();
        long maxFailedNumber = conf.getLong(ShuffleServerConf.LOCAL_STORAGE_INITIALIZE_MAX_FAIL_NUMBER);
        if ((long)failedCount > maxFailedNumber || successCount.get() == 0) {
            throw new RssException(String.format("Initialize %s local storage(s) failed, specified local storage paths size: %s, the conf of %s size: %s", failedCount, localStorageArray.length, ShuffleServerConf.LOCAL_STORAGE_INITIALIZE_MAX_FAIL_NUMBER.key(), maxFailedNumber));
        }
        this.localStorages = Arrays.stream(localStorageArray).filter(Objects::nonNull).collect(Collectors.toList());
        LOG.info("Succeed to initialize storage paths: {}", (Object)StringUtils.join((Object[])new List[]{this.localStorages.stream().map(LocalStorage::getBasePath).collect(Collectors.toList())}));
        this.checker = new LocalStorageChecker(conf, this.localStorages);
    }

    private StorageMedia getStorageTypeForBasePath(String basePath) {
        for (StorageMediaProvider provider : this.typeProviders) {
            StorageMedia result = provider.getStorageMediaFor(basePath);
            if (result == StorageMedia.UNKNOWN) continue;
            return result;
        }
        return StorageMedia.UNKNOWN;
    }

    @Override
    public Storage selectStorage(ShuffleDataFlushEvent event) {
        List candidates;
        String appId = event.getAppId();
        int shuffleId = event.getShuffleId();
        int partitionId = event.getStartPartition();
        LocalStorage storage = this.sortedPartitionsOfStorageMap.get(UnionKey.buildKey((Object[])new Object[]{appId, shuffleId, partitionId}));
        if (storage != null) {
            if (storage.isCorrupted()) {
                if (storage.containsWriteHandler(appId, shuffleId, partitionId)) {
                    LOG.error("LocalStorage: {} is corrupted. Switching another storage for event: {}, some data will be lost", (Object)storage.getBasePath(), (Object)event);
                }
            } else {
                if (event.getUnderStorage() == null) {
                    event.setUnderStorage((Storage)storage);
                }
                return storage;
            }
        }
        if ((candidates = this.localStorages.stream().filter(x -> x.canWrite() && !x.isCorrupted()).collect(Collectors.toList())).size() == 0) {
            return null;
        }
        LocalStorage selectedStorage = (LocalStorage)candidates.get(ShuffleStorageUtils.getStorageIndex((int)candidates.size(), (String)appId, (int)shuffleId, (int)partitionId));
        return (Storage)this.sortedPartitionsOfStorageMap.compute(UnionKey.buildKey((Object[])new Object[]{appId, shuffleId, partitionId}), (key, localStorage) -> {
            if (localStorage == null || localStorage.isCorrupted() || event.getUnderStorage() == null) {
                event.setUnderStorage((Storage)selectedStorage);
                return selectedStorage;
            }
            return localStorage;
        });
    }

    @Override
    public Storage selectStorage(ShuffleDataReadEvent event) {
        String appId = event.getAppId();
        int shuffleId = event.getShuffleId();
        int partitionId = event.getStartPartition();
        return (Storage)this.sortedPartitionsOfStorageMap.get(UnionKey.buildKey((Object[])new Object[]{appId, shuffleId, partitionId}));
    }

    @Override
    public void updateWriteMetrics(ShuffleDataFlushEvent event, long writeTime) {
        super.updateWriteMetrics(event, writeTime);
        ((Counter.Child)ShuffleServerMetrics.counterTotalLocalFileWriteDataSize.labels(new String[]{"ALL"})).inc((double)event.getSize());
        if (event.getUnderStorage() != null) {
            ((Counter.Child)ShuffleServerMetrics.counterTotalLocalFileWriteDataSize.labels(new String[]{event.getUnderStorage().getStoragePath()})).inc((double)event.getSize());
        }
    }

    @Override
    public Checker getStorageChecker() {
        return this.checker;
    }

    @Override
    public void removeResources(PurgeEvent event) {
        String appId = event.getAppId();
        String user = event.getUser();
        List shuffleSet = Optional.ofNullable(event.getShuffleIds()).orElse(Collections.emptyList());
        this.cleanupStorageSelectionCache(event);
        for (LocalStorage storage : this.localStorages) {
            if (event instanceof AppPurgeEvent) {
                storage.removeHandlers(appId);
            }
            for (Integer shuffleId : shuffleSet) {
                storage.removeResources(RssUtils.generateShuffleKey((String)appId, (int)shuffleId));
            }
        }
        ShuffleDeleteHandler deleteHandler = ShuffleHandlerFactory.getInstance().createShuffleDeleteHandler(new CreateShuffleDeleteHandlerRequest(StorageType.LOCALFILE.name(), new Configuration()));
        List<String> deletePaths = this.storageBasePaths.stream().flatMap(path -> {
            String basicPath = ShuffleStorageUtils.getFullShuffleDataFolder((String)path, (String)appId);
            if (event instanceof ShufflePurgeEvent) {
                ArrayList<String> paths = new ArrayList<String>();
                Iterator iterator = shuffleSet.iterator();
                while (iterator.hasNext()) {
                    int shuffleId = (Integer)iterator.next();
                    paths.add(ShuffleStorageUtils.getFullShuffleDataFolder((String)basicPath, (String)String.valueOf(shuffleId)));
                }
                return paths.stream();
            }
            return Stream.of(basicPath);
        }).collect(Collectors.toList());
        deleteHandler.delete(deletePaths.toArray(new String[deletePaths.size()]), appId, user);
    }

    private void cleanupStorageSelectionCache(PurgeEvent event) {
        Function<String, Boolean> deleteConditionFunc = null;
        String prefixKey = null;
        if (event instanceof AppPurgeEvent) {
            prefixKey = UnionKey.buildKey((Object[])new Object[]{event.getAppId(), ""});
            deleteConditionFunc = partitionUnionKey -> UnionKey.startsWith((String)partitionUnionKey, (Object[])new Object[]{event.getAppId()});
        } else if (event instanceof ShufflePurgeEvent) {
            int shuffleId = event.getShuffleIds().get(0);
            prefixKey = UnionKey.buildKey((Object[])new Object[]{event.getAppId(), shuffleId, ""});
            deleteConditionFunc = partitionUnionKey -> UnionKey.startsWith((String)partitionUnionKey, (Object[])new Object[]{event.getAppId(), shuffleId});
        }
        if (prefixKey == null) {
            throw new RssException("Prefix key is null when handles event: " + event);
        }
        long startTime = System.currentTimeMillis();
        this.deleteElement(this.sortedPartitionsOfStorageMap.tailMap((Object)prefixKey), deleteConditionFunc);
        LOG.info("Cleaning the storage selection cache costs: {}(ms) for event: {}", (Object)(System.currentTimeMillis() - startTime), (Object)event);
    }

    private <K, V> void deleteElement(Map<K, V> sortedPartitionsOfStorageMap, Function<K, Boolean> deleteConditionFunc) {
        Map.Entry<K, V> entry;
        Iterator<Map.Entry<K, V>> iterator = sortedPartitionsOfStorageMap.entrySet().iterator();
        while (iterator.hasNext() && deleteConditionFunc.apply((entry = iterator.next()).getKey()).booleanValue()) {
            iterator.remove();
        }
    }

    @Override
    public void registerRemoteStorage(String appId, RemoteStorageInfo remoteStorageInfo) {
    }

    @Override
    public void checkAndClearLeakedShuffleData(Collection<String> appIds) {
        HashSet appIdsOnStorages = new HashSet();
        for (LocalStorage localStorage : this.localStorages) {
            if (localStorage.isCorrupted()) continue;
            Set appIdsOnStorage = localStorage.getAppIds();
            appIdsOnStorages.addAll(appIdsOnStorage);
        }
        for (String appId : appIdsOnStorages) {
            if (appIds.contains(appId)) continue;
            ShuffleDeleteHandler deleteHandler = ShuffleHandlerFactory.getInstance().createShuffleDeleteHandler(new CreateShuffleDeleteHandlerRequest(StorageType.LOCALFILE.name(), new Configuration()));
            String[] deletePaths = new String[this.storageBasePaths.size()];
            for (int i = 0; i < this.storageBasePaths.size(); ++i) {
                deletePaths[i] = ShuffleStorageUtils.getFullShuffleDataFolder((String)this.storageBasePaths.get(i), (String)appId);
            }
            deleteHandler.delete(deletePaths, appId, UNKNOWN_USER_NAME);
        }
    }

    @Override
    public Map<String, StorageInfo> getStorageInfo() {
        HashMap<String, StorageInfo> result = Maps.newHashMap();
        for (LocalStorage storage : this.localStorages) {
            String mountPoint = storage.getMountPoint();
            long capacity = storage.getCapacity();
            long wroteBytes = storage.getServiceUsedBytes();
            StorageStatus status = StorageStatus.NORMAL;
            if (storage.isCorrupted()) {
                status = StorageStatus.UNHEALTHY;
            } else if (!storage.canWrite()) {
                status = StorageStatus.OVERUSED;
            }
            StorageMedia media = storage.getStorageMedia();
            if (media == null) {
                media = StorageMedia.UNKNOWN;
            }
            StorageInfo info = new StorageInfo(mountPoint, media, capacity, wroteBytes, status);
            result.put(mountPoint, info);
        }
        return result;
    }

    public List<LocalStorage> getStorages() {
        return this.localStorages;
    }

    @VisibleForTesting
    public Map<String, LocalStorage> getSortedPartitionsOfStorageMap() {
        return this.sortedPartitionsOfStorageMap;
    }
}

