/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import com.google.common.annotations.VisibleForTesting;
import io.netty.buffer.ByteBuf;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.mledger.AsyncCallbacks;
import org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.bookkeeper.mledger.ManagedLedger;
import org.apache.bookkeeper.mledger.ManagedLedgerException;
import org.apache.bookkeeper.mledger.ManagedLedgerReplayTask;
import org.apache.bookkeeper.mledger.Position;
import org.apache.bookkeeper.mledger.util.ManagedLedgerUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.service.Producer;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.common.api.proto.KeyValue;
import org.apache.pulsar.common.api.proto.MessageMetadata;
import org.apache.pulsar.common.protocol.Commands;
import org.apache.pulsar.common.protocol.Markers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageDeduplication {
    private final PulsarService pulsar;
    private final PersistentTopic topic;
    private final ManagedLedger managedLedger;
    private final ManagedLedgerReplayTask replayTask;
    private ManagedCursor managedCursor;
    private static final String IS_LAST_CHUNK = "isLastChunk";
    private volatile Status status;
    @VisibleForTesting
    final Map<String, Long> highestSequencedPushed = new ConcurrentHashMap<String, Long>();
    @VisibleForTesting
    final Map<String, Long> highestSequencedPersisted = new ConcurrentHashMap<String, Long>();
    private final int snapshotInterval;
    private int snapshotCounter;
    private volatile long lastSnapshotTimestamp = 0L;
    private final int maxNumberOfProducers;
    private final Map<String, Long> inactiveProducers = new ConcurrentHashMap<String, Long>();
    private final String replicatorPrefix;
    private final AtomicBoolean snapshotTaking = new AtomicBoolean(false);
    private static final Logger log = LoggerFactory.getLogger(MessageDeduplication.class);

    public MessageDeduplication(PulsarService pulsar, PersistentTopic topic, ManagedLedger managedLedger) {
        this.pulsar = pulsar;
        this.topic = topic;
        this.managedLedger = managedLedger;
        this.status = Status.Initialized;
        this.snapshotInterval = pulsar.getConfiguration().getBrokerDeduplicationEntriesInterval();
        this.maxNumberOfProducers = pulsar.getConfiguration().getBrokerDeduplicationMaxNumberOfProducers();
        this.snapshotCounter = 0;
        this.replicatorPrefix = pulsar.getConfiguration().getReplicatorPrefix();
        this.replayTask = new ManagedLedgerReplayTask("MessageDeduplication", (Executor)pulsar.getExecutor(), 100);
    }

    public Status getStatus() {
        return this.status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<Void> checkStatus() {
        boolean shouldBeEnabled = this.topic.isDeduplicationEnabled();
        MessageDeduplication messageDeduplication = this;
        synchronized (messageDeduplication) {
            if (this.status == Status.Recovering || this.status == Status.Removing) {
                this.pulsar.getExecutor().schedule(this::checkStatus, 1L, TimeUnit.MINUTES);
                return CompletableFuture.completedFuture(null);
            }
            if (this.status == Status.Initialized && !shouldBeEnabled) {
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        log.info("[{}] Deleted deduplication cursor", (Object)MessageDeduplication.this.topic.getName());
                    }

                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                        } else {
                            log.error("[{}] Deleted deduplication cursor error", (Object)MessageDeduplication.this.topic.getName(), (Object)exception);
                        }
                    }
                }, null);
            }
            if (this.status == Status.Enabled && !shouldBeEnabled) {
                final CompletableFuture<Void> future = new CompletableFuture<Void>();
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        MessageDeduplication.this.managedCursor = null;
                        MessageDeduplication.this.highestSequencedPushed.clear();
                        MessageDeduplication.this.highestSequencedPersisted.clear();
                        future.complete(null);
                        log.info("[{}] Disabled deduplication", (Object)MessageDeduplication.this.topic.getName());
                    }

                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                            MessageDeduplication.this.managedCursor = null;
                            MessageDeduplication.this.highestSequencedPushed.clear();
                            MessageDeduplication.this.highestSequencedPersisted.clear();
                            future.complete(null);
                        } else {
                            log.warn("[{}] Failed to disable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)exception.getMessage());
                            MessageDeduplication.this.status = Status.Failed;
                            future.completeExceptionally(exception);
                        }
                    }
                }, null);
                return future;
            }
            if ((this.status == Status.Disabled || this.status == Status.Initialized) && shouldBeEnabled) {
                CompletionStage future = ManagedLedgerUtils.openCursor((ManagedLedger)this.managedLedger, (String)"pulsar.dedup").thenCompose(this::replayCursor);
                ((CompletableFuture)future).exceptionally(arg_0 -> this.lambda$checkStatus$0((CompletableFuture)future, arg_0));
                return future;
            }
            return CompletableFuture.completedFuture(null);
        }
    }

    private CompletableFuture<Void> replayCursor(ManagedCursor cursor) {
        this.managedCursor = cursor;
        this.managedCursor.getProperties().forEach((k, v) -> {
            this.producerRemoved((String)k);
            this.highestSequencedPushed.put((String)k, (Long)v);
            this.highestSequencedPersisted.put((String)k, (Long)v);
        });
        log.info("[{}] Replaying {} entries for deduplication", (Object)this.topic.getName(), (Object)this.managedCursor.getNumberOfEntries());
        return ((CompletableFuture)this.replayTask.replay(cursor, (__, buffer) -> {
            MessageMetadata metadata = Commands.parseMessageMetadata((ByteBuf)buffer);
            String producerName = metadata.getProducerName();
            long sequenceId = Math.max(metadata.getHighestSequenceId(), metadata.getSequenceId());
            this.highestSequencedPushed.put(producerName, sequenceId);
            this.highestSequencedPersisted.put(producerName, sequenceId);
            this.producerRemoved(producerName);
        }).thenCompose(optPosition -> {
            if (optPosition.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            this.snapshotCounter = this.replayTask.getNumEntriesProcessed();
            if (this.snapshotCounter >= this.snapshotInterval) {
                return this.takeSnapshot((Position)optPosition.get());
            }
            return CompletableFuture.completedFuture(null);
        })).thenRun(() -> {
            this.status = Status.Enabled;
            log.info("[{}] Enabled deduplication", (Object)this.topic.getName());
        });
    }

    public boolean isEnabled() {
        return this.status == Status.Enabled;
    }

    public MessageDupStatus isDuplicate(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        this.setContextPropsIfRepl(publishContext, headersAndPayload);
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return MessageDupStatus.NotDup;
        }
        if (Producer.isRemoteOrShadow(publishContext.getProducerName(), this.replicatorPrefix)) {
            if (!publishContext.supportsReplDedupByLidAndEid()) {
                return this.isDuplicateReplV1(publishContext, headersAndPayload);
            }
            return this.isDuplicateReplV2(publishContext, headersAndPayload);
        }
        return this.isDuplicateNormal(publishContext, headersAndPayload, false);
    }

    public MessageDupStatus isDuplicateReplV1(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        int readerIndex = headersAndPayload.readerIndex();
        MessageMetadata md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
        headersAndPayload.readerIndex(readerIndex);
        String producerName = md.getProducerName();
        long sequenceId = md.getSequenceId();
        long highestSequenceId = Math.max(md.getHighestSequenceId(), sequenceId);
        publishContext.setOriginalProducerName(producerName);
        publishContext.setOriginalSequenceId(sequenceId);
        publishContext.setOriginalHighestSequenceId(highestSequenceId);
        return this.isDuplicateNormal(publishContext, headersAndPayload, true);
    }

    private void setContextPropsIfRepl(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        if (publishContext.isMarkerMessage()) {
            MessageMetadata md = Commands.peekMessageMetadata((ByteBuf)headersAndPayload, (String)"Check-Deduplicate", (long)-1L);
            if (md != null && md.hasMarkerType() && Markers.isReplicationMarker((int)md.getMarkerType())) {
                publishContext.setProperty("__MSG_PROP_IS_REPL_MARKER", "");
            }
            return;
        }
        if (Producer.isRemoteOrShadow(publishContext.getProducerName(), this.replicatorPrefix)) {
            int readerIndex = headersAndPayload.readerIndex();
            MessageMetadata md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
            headersAndPayload.readerIndex(readerIndex);
            List kvPairList = md.getPropertiesList();
            for (KeyValue kvPair : kvPairList) {
                if (!kvPair.getKey().equals("__MSG_PROP_REPL_SOURCE_POSITION")) continue;
                if (!kvPair.getValue().contains(":")) {
                    log.warn("[{}] Unexpected {}: {}", new Object[]{publishContext.getProducerName(), "__MSG_PROP_REPL_SOURCE_POSITION", kvPair.getValue()});
                    break;
                }
                String[] ledgerIdAndEntryId = kvPair.getValue().split(":");
                if (ledgerIdAndEntryId.length != 2 || !StringUtils.isNumeric((CharSequence)ledgerIdAndEntryId[0]) || !StringUtils.isNumeric((CharSequence)ledgerIdAndEntryId[1])) {
                    log.warn("[{}] Unexpected {}: {}", new Object[]{publishContext.getProducerName(), "__MSG_PROP_REPL_SOURCE_POSITION", kvPair.getValue()});
                    break;
                }
                long[] positionPair = new long[]{Long.valueOf(ledgerIdAndEntryId[0]), Long.valueOf(ledgerIdAndEntryId[1])};
                publishContext.setProperty("__MSG_PROP_REPL_SOURCE_POSITION", positionPair);
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MessageDupStatus isDuplicateReplV2(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        Object positionPairObj = publishContext.getProperty("__MSG_PROP_REPL_SOURCE_POSITION");
        if (positionPairObj == null || !(positionPairObj instanceof long[])) {
            log.error("[{}] Message can not determine whether the message is duplicated due to the acquired messages props were are invalid. producer={}. supportsReplDedupByLidAndEid: {}, sequence-id {}, prop-{}: not in expected format", new Object[]{this.topic.getName(), publishContext.getProducerName(), publishContext.supportsReplDedupByLidAndEid(), publishContext.getSequenceId(), "__MSG_PROP_REPL_SOURCE_POSITION"});
            return MessageDupStatus.Unknown;
        }
        long[] positionPair = (long[])positionPairObj;
        long replSequenceLId = positionPair[0];
        long replSequenceEId = positionPair[1];
        String lastSequenceLIdKey = publishContext.getProducerName() + "_LID";
        String lastSequenceEIdKey = publishContext.getProducerName() + "_EID";
        Map<String, Long> map = this.highestSequencedPushed;
        synchronized (map) {
            Long lastSequenceLIdPushed = this.highestSequencedPushed.get(lastSequenceLIdKey);
            Long lastSequenceEIdPushed = this.highestSequencedPushed.get(lastSequenceEIdKey);
            if (lastSequenceLIdPushed != null && lastSequenceEIdPushed != null && (replSequenceLId < lastSequenceLIdPushed || replSequenceLId == lastSequenceLIdPushed && replSequenceEId <= lastSequenceEIdPushed)) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={}. publishing {}:{}, latest publishing in-progress {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), lastSequenceLIdPushed, lastSequenceEIdPushed, lastSequenceLIdPushed, lastSequenceEIdPushed});
                }
                Long lastSequenceLIdPersisted = this.highestSequencedPersisted.get(lastSequenceLIdKey);
                Long lastSequenceEIdPersisted = this.highestSequencedPersisted.get(lastSequenceEIdKey);
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={}. publishing {}:{}, latest persisted {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), replSequenceLId, replSequenceEId, lastSequenceLIdPersisted, lastSequenceEIdPersisted});
                }
                if (lastSequenceLIdPersisted != null && lastSequenceEIdPersisted != null && (replSequenceLId < lastSequenceLIdPersisted || replSequenceLId == lastSequenceLIdPersisted && replSequenceEId <= lastSequenceEIdPersisted)) {
                    return MessageDupStatus.Dup;
                }
                return MessageDupStatus.Unknown;
            }
            this.highestSequencedPushed.put(lastSequenceLIdKey, replSequenceLId);
            this.highestSequencedPushed.put(lastSequenceEIdKey, replSequenceEId);
        }
        if (log.isDebugEnabled()) {
            log.debug("[{}] Message identified as non-duplicated producer={}. publishing {}:{}", new Object[]{this.topic.getName(), publishContext.getProducerName(), replSequenceLId, replSequenceEId});
        }
        return MessageDupStatus.NotDup;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MessageDupStatus isDuplicateNormal(Topic.PublishContext publishContext, ByteBuf headersAndPayload, boolean useOriginalProducerName) {
        String producerName = publishContext.getProducerName();
        if (useOriginalProducerName) {
            producerName = publishContext.getOriginalProducerName();
        }
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = Math.max(publishContext.getHighestSequenceId(), sequenceId);
        MessageMetadata md = null;
        long chunkID = -1L;
        long totalChunk = -1L;
        if (publishContext.isChunked()) {
            if (md == null) {
                int readerIndex = headersAndPayload.readerIndex();
                md = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
                headersAndPayload.readerIndex(readerIndex);
            }
            chunkID = md.getChunkId();
            totalChunk = md.getNumChunksFromMsg();
        }
        if (chunkID != -1L && chunkID != totalChunk - 1L) {
            publishContext.setProperty(IS_LAST_CHUNK, Boolean.FALSE);
            return MessageDupStatus.NotDup;
        }
        Map<String, Long> map = this.highestSequencedPushed;
        synchronized (map) {
            Long lastSequenceIdPushed = this.highestSequencedPushed.get(producerName);
            if (lastSequenceIdPushed != null && sequenceId <= lastSequenceIdPushed) {
                Long lastSequenceIdPersisted;
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={} seq-id={} -- highest-seq-id={}", new Object[]{this.topic.getName(), producerName, sequenceId, lastSequenceIdPushed});
                }
                if ((lastSequenceIdPersisted = this.highestSequencedPersisted.get(producerName)) != null && sequenceId <= lastSequenceIdPersisted) {
                    return MessageDupStatus.Dup;
                }
                return MessageDupStatus.Unknown;
            }
            this.highestSequencedPushed.put(producerName, highestSequenceId);
        }
        if (chunkID != -1L && chunkID == totalChunk - 1L) {
            publishContext.setProperty(IS_LAST_CHUNK, Boolean.TRUE);
        }
        return MessageDupStatus.NotDup;
    }

    public void recordMessagePersisted(Topic.PublishContext publishContext, Position position) {
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return;
        }
        if (publishContext.getProducerName().startsWith(this.replicatorPrefix) && publishContext.supportsReplDedupByLidAndEid()) {
            this.recordMessagePersistedRepl(publishContext, position);
        } else {
            this.recordMessagePersistedNormal(publishContext, position);
        }
    }

    public void recordMessagePersistedRepl(Topic.PublishContext publishContext, Position position) {
        Object positionPairObj = publishContext.getProperty("__MSG_PROP_REPL_SOURCE_POSITION");
        if (positionPairObj == null || !(positionPairObj instanceof long[])) {
            log.error("[{}] Can not persist highest sequence-id due to the acquired messages props are invalid. producer={}. supportsReplDedupByLidAndEid: {}, sequence-id {}, prop-{}: not in expected format", new Object[]{this.topic.getName(), publishContext.getProducerName(), publishContext.supportsReplDedupByLidAndEid(), publishContext.getSequenceId(), "__MSG_PROP_REPL_SOURCE_POSITION"});
            this.recordMessagePersistedNormal(publishContext, position);
            return;
        }
        long[] positionPair = (long[])positionPairObj;
        long replSequenceLId = positionPair[0];
        long replSequenceEId = positionPair[1];
        String lastSequenceLIdKey = publishContext.getProducerName() + "_LID";
        String lastSequenceEIdKey = publishContext.getProducerName() + "_EID";
        this.highestSequencedPersisted.put(lastSequenceLIdKey, replSequenceLId);
        this.highestSequencedPersisted.put(lastSequenceEIdKey, replSequenceEId);
        this.increaseSnapshotCounterAndTakeSnapshotIfNeeded(position);
    }

    public void recordMessagePersistedNormal(Topic.PublishContext publishContext, Position position) {
        Boolean isLastChunk;
        String producerName = publishContext.getProducerName();
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = publishContext.getHighestSequenceId();
        if (publishContext.getOriginalProducerName() != null) {
            producerName = publishContext.getOriginalProducerName();
            sequenceId = publishContext.getOriginalSequenceId();
            highestSequenceId = publishContext.getOriginalHighestSequenceId();
        }
        if ((isLastChunk = (Boolean)publishContext.getProperty(IS_LAST_CHUNK)) == null || isLastChunk.booleanValue()) {
            this.highestSequencedPersisted.put(producerName, Math.max(highestSequenceId, sequenceId));
        }
        this.increaseSnapshotCounterAndTakeSnapshotIfNeeded(position);
    }

    private void increaseSnapshotCounterAndTakeSnapshotIfNeeded(Position position) {
        if (++this.snapshotCounter >= this.snapshotInterval) {
            this.snapshotCounter = 0;
            this.takeSnapshot(position);
        } else if (log.isDebugEnabled()) {
            log.debug("[{}] Waiting for sequence-id snapshot {}/{}", new Object[]{this.topic.getName(), this.snapshotCounter, this.snapshotInterval});
        }
    }

    public void resetHighestSequenceIdPushed() {
        if (!this.isEnabled()) {
            return;
        }
        this.highestSequencedPushed.clear();
        for (String producer : this.highestSequencedPersisted.keySet()) {
            this.highestSequencedPushed.put(producer, this.highestSequencedPersisted.get(producer));
        }
    }

    private CompletableFuture<Void> takeSnapshot(Position position) {
        if (log.isDebugEnabled()) {
            log.debug("[{}] Taking snapshot of sequence ids map", (Object)this.topic.getName());
        }
        if (!this.snapshotTaking.compareAndSet(false, true)) {
            log.warn("[{}] There is a pending snapshot when taking snapshot for {}", (Object)this.topic.getName(), (Object)position);
            return CompletableFuture.completedFuture(null);
        }
        TreeMap snapshot = new TreeMap();
        this.highestSequencedPersisted.forEach((producerName, sequenceId) -> {
            if (snapshot.size() < this.maxNumberOfProducers) {
                snapshot.put(producerName, sequenceId);
            }
        });
        ManagedCursor cursor = this.managedCursor;
        if (cursor == null) {
            log.warn("[{}] Cursor is null when taking snapshot for {}", (Object)this.topic.getName(), (Object)position);
            return CompletableFuture.completedFuture(null);
        }
        CompletionStage future = ManagedLedgerUtils.markDelete((ManagedCursor)cursor, (Position)position, snapshot).thenRun(() -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Stored new deduplication snapshot at {}", (Object)this.topic.getName(), (Object)position);
            }
            this.lastSnapshotTimestamp = System.currentTimeMillis();
            this.snapshotTaking.set(false);
        });
        ((CompletableFuture)future).exceptionally(e -> {
            log.warn("[{}] Failed to store new deduplication snapshot at {}", new Object[]{this.topic.getName(), position, e});
            this.snapshotTaking.set(false);
            return null;
        });
        return future;
    }

    public void producerAdded(String producerName) {
        if (!this.isEnabled()) {
            return;
        }
        this.inactiveProducers.remove(producerName);
    }

    public void producerRemoved(String producerName) {
        if (!this.isEnabled()) {
            return;
        }
        this.inactiveProducers.put(producerName, System.currentTimeMillis());
    }

    public synchronized void purgeInactiveProducers() {
        long minimumActiveTimestamp = System.currentTimeMillis() - TimeUnit.MINUTES.toMillis(this.pulsar.getConfiguration().getBrokerDeduplicationProducerInactivityTimeoutMinutes());
        if (!this.isEnabled()) {
            if (!this.inactiveProducers.isEmpty()) {
                this.inactiveProducers.clear();
            }
            return;
        }
        Iterator<Map.Entry<String, Long>> mapIterator = this.inactiveProducers.entrySet().iterator();
        boolean hasInactive = false;
        while (mapIterator.hasNext()) {
            Map.Entry<String, Long> entry = mapIterator.next();
            String producerName = entry.getKey();
            long lastActiveTimestamp = entry.getValue();
            if (lastActiveTimestamp >= minimumActiveTimestamp) continue;
            log.info("[{}] Purging dedup information for producer {}", (Object)this.topic.getName(), (Object)producerName);
            mapIterator.remove();
            this.highestSequencedPushed.remove(producerName);
            this.highestSequencedPersisted.remove(producerName);
            hasInactive = true;
        }
        if (hasInactive && this.isEnabled()) {
            this.takeSnapshot(this.getManagedCursor().getMarkDeletedPosition());
        }
    }

    public long getLastPublishedSequenceId(String producerName) {
        Long sequenceId = this.highestSequencedPushed.get(producerName);
        return sequenceId != null ? sequenceId : -1L;
    }

    public void takeSnapshot() {
        if (!this.isEnabled()) {
            return;
        }
        Integer interval = (Integer)this.topic.getHierarchyTopicPolicies().getDeduplicationSnapshotIntervalSeconds().get();
        long currentTimeStamp = System.currentTimeMillis();
        if (interval == null || interval <= 0 || currentTimeStamp - this.lastSnapshotTimestamp < TimeUnit.SECONDS.toMillis(interval.intValue())) {
            return;
        }
        Position position = this.managedLedger.getLastConfirmedEntry();
        if (position == null) {
            return;
        }
        Position markDeletedPosition = this.managedCursor.getMarkDeletedPosition();
        if (markDeletedPosition != null && position.compareTo(markDeletedPosition) <= 0) {
            return;
        }
        this.takeSnapshot(position);
    }

    @VisibleForTesting
    ManagedCursor getManagedCursor() {
        return this.managedCursor;
    }

    @VisibleForTesting
    Map<String, Long> getInactiveProducers() {
        return this.inactiveProducers;
    }

    private /* synthetic */ Void lambda$checkStatus$0(CompletableFuture future, Throwable e) {
        this.status = Status.Failed;
        log.error("[{}] Failed to enable deduplication", (Object)this.topic.getName(), (Object)e);
        future.completeExceptionally(e);
        return null;
    }

    static enum Status {
        Initialized,
        Disabled,
        Recovering,
        Enabled,
        Removing,
        Failed;

    }

    @VisibleForTesting
    public static enum MessageDupStatus {
        Unknown,
        NotDup,
        Dup;

    }

    public static class MessageDupUnknownException
    extends RuntimeException {
        public MessageDupUnknownException(String topicName, String producerName) {
            super(String.format("[%s][%s]Cannot determine whether the message is a duplicate at this time", topicName, producerName));
        }
    }
}

