/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.function.IntFunction;
import java.util.zip.CRC32;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSDataOutputStreamBuilder;
import org.apache.hadoop.fs.FSInputChecker;
import org.apache.hadoop.fs.FSOutputSummer;
import org.apache.hadoop.fs.FileRange;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.FutureDataInputStreamBuilder;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.VectoredReadUtils;
import org.apache.hadoop.fs.impl.AbstractFSBuilderImpl;
import org.apache.hadoop.fs.impl.CombinedFileRange;
import org.apache.hadoop.fs.impl.OpenFileParameters;
import org.apache.hadoop.fs.impl.PathCapabilitiesSupport;
import org.apache.hadoop.fs.impl.StoreImplementationUtils;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.fs.statistics.IOStatisticsSupport;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.LambdaUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Progressable;

@InterfaceAudience.Public
@InterfaceStability.Stable
public abstract class ChecksumFileSystem
extends FilterFileSystem {
    private static final byte[] CHECKSUM_VERSION = new byte[]{99, 114, 99, 0};
    private int bytesPerChecksum = 512;
    private boolean verifyChecksum = true;
    private boolean writeChecksum = true;
    private static final PathFilter DEFAULT_FILTER = new PathFilter(){

        @Override
        public boolean accept(Path file) {
            return !ChecksumFileSystem.isChecksumFile(file);
        }
    };

    public static double getApproxChkSumLength(long size) {
        return 0.01f * (float)size;
    }

    public ChecksumFileSystem(FileSystem fs) {
        super(fs);
    }

    @Override
    public void setConf(Configuration conf) {
        super.setConf(conf);
        if (conf != null) {
            this.bytesPerChecksum = conf.getInt("file.bytes-per-checksum", 512);
            Preconditions.checkState(this.bytesPerChecksum > 0, "bytes per checksum should be positive but was %s", this.bytesPerChecksum);
        }
    }

    @Override
    public void setVerifyChecksum(boolean verifyChecksum) {
        this.verifyChecksum = verifyChecksum;
    }

    @Override
    public void setWriteChecksum(boolean writeChecksum) {
        this.writeChecksum = writeChecksum;
    }

    @Override
    public FileSystem getRawFileSystem() {
        return this.fs;
    }

    public Path getChecksumFile(Path file) {
        return new Path(file.getParent(), "." + file.getName() + ".crc");
    }

    public static boolean isChecksumFile(Path file) {
        String name = file.getName();
        return name.startsWith(".") && name.endsWith(".crc");
    }

    public long getChecksumFileLength(Path file, long fileSize) {
        return ChecksumFileSystem.getChecksumLength(fileSize, this.getBytesPerSum());
    }

    public int getBytesPerSum() {
        return this.bytesPerChecksum;
    }

    private int getSumBufferSize(int bytesPerSum, int bufferSize) {
        int defaultBufferSize = this.getConf().getInt("file.stream-buffer-size", 4096);
        int proportionalBufferSize = bufferSize / bytesPerSum;
        return Math.max(bytesPerSum, Math.max(proportionalBufferSize, defaultBufferSize));
    }

    @Override
    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        InputStream in;
        FileSystem fs;
        if (this.verifyChecksum) {
            fs = this;
            in = new ChecksumFSInputChecker(this, f, bufferSize);
        } else {
            fs = this.getRawFileSystem();
            in = fs.open(f, bufferSize);
        }
        return new FSDataBoundedInputStream(fs, f, in);
    }

    @Override
    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new UnsupportedOperationException("Append is not supported by ChecksumFileSystem");
    }

    @Override
    public boolean truncate(Path f, long newLength) throws IOException {
        throw new UnsupportedOperationException("Truncate is not supported by ChecksumFileSystem");
    }

    @Override
    public void concat(Path f, Path[] psrcs) throws IOException {
        throw new UnsupportedOperationException("Concat is not supported by ChecksumFileSystem");
    }

    public static long getChecksumLength(long size, int bytesPerSum) {
        return (size + (long)bytesPerSum - 1L) / (long)bytesPerSum * 4L + 8L;
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(f, permission, overwrite, true, bufferSize, replication, blockSize, progress);
    }

    private FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, boolean createParent, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        FSDataOutputStream out;
        Path parent = f.getParent();
        if (parent != null) {
            if (!createParent && !this.exists(parent)) {
                throw new FileNotFoundException("Parent directory doesn't exist: " + parent);
            }
            if (!this.mkdirs(parent)) {
                throw new IOException("Mkdirs failed to create " + parent + " (exists=" + this.exists(parent) + ", cwd=" + this.getWorkingDirectory() + ")");
            }
        }
        if (this.writeChecksum) {
            out = new FSDataOutputStream(new ChecksumFSOutputSummer(this, f, overwrite, bufferSize, replication, blockSize, progress, permission), null);
        } else {
            out = this.fs.create(f, permission, overwrite, bufferSize, replication, blockSize, progress);
            Path checkFile = this.getChecksumFile(f);
            if (this.fs.exists(checkFile)) {
                this.fs.delete(checkFile, true);
            }
        }
        return out;
    }

    @Override
    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(f, permission, overwrite, false, bufferSize, replication, blockSize, progress);
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress, Options.ChecksumOpt checksumOpt) throws IOException {
        return this.create(f, permission, flags.contains((Object)CreateFlag.OVERWRITE), bufferSize, replication, blockSize, progress);
    }

    @Override
    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(f, permission, flags.contains((Object)CreateFlag.OVERWRITE), false, bufferSize, replication, blockSize, progress);
    }

    @Override
    public void setPermission(Path src, final FsPermission permission) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.setPermission(p, permission);
                return true;
            }
        }.run(src);
    }

    @Override
    public void setOwner(Path src, final String username, final String groupname) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.setOwner(p, username, groupname);
                return true;
            }
        }.run(src);
    }

    @Override
    public void setAcl(Path src, final List<AclEntry> aclSpec) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.setAcl(p, aclSpec);
                return true;
            }
        }.run(src);
    }

    @Override
    public void modifyAclEntries(Path src, final List<AclEntry> aclSpec) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.modifyAclEntries(p, aclSpec);
                return true;
            }
        }.run(src);
    }

    @Override
    public void removeAcl(Path src) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.removeAcl(p);
                return true;
            }
        }.run(src);
    }

    @Override
    public void removeAclEntries(Path src, final List<AclEntry> aclSpec) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.removeAclEntries(p, aclSpec);
                return true;
            }
        }.run(src);
    }

    @Override
    public void removeDefaultAcl(Path src) throws IOException {
        new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                ChecksumFileSystem.this.fs.removeDefaultAcl(p);
                return true;
            }
        }.run(src);
    }

    @Override
    public boolean setReplication(Path src, final short replication) throws IOException {
        return new FsOperation(){

            @Override
            boolean apply(Path p) throws IOException {
                return ChecksumFileSystem.this.fs.setReplication(p, replication);
            }
        }.run(src);
    }

    @Override
    public boolean rename(Path src, Path dst) throws IOException {
        boolean value;
        if (this.fs.isDirectory(src)) {
            return this.fs.rename(src, dst);
        }
        if (this.fs.isDirectory(dst)) {
            dst = new Path(dst, src.getName());
        }
        if (!(value = this.fs.rename(src, dst))) {
            return false;
        }
        Path srcCheckFile = this.getChecksumFile(src);
        Path dstCheckFile = this.getChecksumFile(dst);
        if (this.fs.exists(srcCheckFile)) {
            value = this.fs.rename(srcCheckFile, dstCheckFile);
        } else if (this.fs.exists(dstCheckFile)) {
            value = this.fs.delete(dstCheckFile, true);
        }
        return value;
    }

    @Override
    public boolean delete(Path f, boolean recursive) throws IOException {
        FileStatus fstatus = null;
        try {
            fstatus = this.fs.getFileStatus(f);
        }
        catch (FileNotFoundException e) {
            return false;
        }
        if (fstatus.isDirectory()) {
            return this.fs.delete(f, recursive);
        }
        Path checkFile = this.getChecksumFile(f);
        if (this.fs.exists(checkFile)) {
            this.fs.delete(checkFile, true);
        }
        return this.fs.delete(f, true);
    }

    @Override
    public FileStatus[] listStatus(Path f) throws IOException {
        return this.fs.listStatus(f, DEFAULT_FILTER);
    }

    @Override
    public RemoteIterator<FileStatus> listStatusIterator(Path p) throws IOException {
        return new FileSystem.DirListingIterator<FileStatus>(p);
    }

    @Override
    public RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f) throws IOException {
        return this.fs.listLocatedStatus(f, DEFAULT_FILTER);
    }

    @Override
    public boolean mkdirs(Path f) throws IOException {
        return this.fs.mkdirs(f);
    }

    @Override
    public void copyFromLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        Configuration conf = this.getConf();
        FileUtil.copy(ChecksumFileSystem.getLocal(conf), src, this, dst, delSrc, conf);
    }

    @Override
    public void copyToLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        Configuration conf = this.getConf();
        FileUtil.copy(this, src, ChecksumFileSystem.getLocal(conf), dst, delSrc, conf);
    }

    public void copyToLocalFile(Path src, Path dst, boolean copyCrc) throws IOException {
        if (!this.fs.isDirectory(src)) {
            this.fs.copyToLocalFile(src, dst);
            FileSystem localFs = ChecksumFileSystem.getLocal(this.getConf()).getRawFileSystem();
            if (localFs.isDirectory(dst)) {
                dst = new Path(dst, src.getName());
            }
            if (localFs.exists(dst = this.getChecksumFile(dst))) {
                localFs.delete(dst, true);
            }
            Path checksumFile = this.getChecksumFile(src);
            if (copyCrc && this.fs.exists(checksumFile)) {
                this.fs.copyToLocalFile(checksumFile, dst);
            }
        } else {
            FileStatus[] srcs;
            for (FileStatus srcFile : srcs = this.listStatus(src)) {
                this.copyToLocalFile(srcFile.getPath(), new Path(dst, srcFile.getPath().getName()), copyCrc);
            }
        }
    }

    @Override
    public Path startLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        return tmpLocalFile;
    }

    @Override
    public void completeLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        this.moveFromLocalFile(tmpLocalFile, fsOutputFile);
    }

    public boolean reportChecksumFailure(Path f, FSDataInputStream in, long inPos, FSDataInputStream sums, long sumsPos) {
        return false;
    }

    @Override
    public FutureDataInputStreamBuilder openFile(Path path) throws IOException, UnsupportedOperationException {
        return ChecksumFileSystem.createDataInputStreamBuilder((FileSystem)this, path).getThisBuilder();
    }

    @Override
    protected CompletableFuture<FSDataInputStream> openFileWithOptions(Path path, OpenFileParameters parameters) throws IOException {
        AbstractFSBuilderImpl.rejectUnknownMandatoryKeys(parameters.getMandatoryKeys(), Options.OpenFileOptions.FS_OPTION_OPENFILE_STANDARD_OPTIONS, "for " + path);
        return LambdaUtils.eval(new CompletableFuture(), () -> this.open(path, parameters.getBufferSize()));
    }

    @Override
    public FSDataOutputStreamBuilder createFile(Path path) {
        return ((FSDataOutputStreamBuilder)ChecksumFileSystem.createDataOutputStreamBuilder(this, path).create()).overwrite(true);
    }

    @Override
    public FSDataOutputStreamBuilder appendFile(Path path) {
        return ChecksumFileSystem.createDataOutputStreamBuilder(this, path).append();
    }

    @Override
    public boolean hasPathCapability(Path path, String capability) throws IOException {
        Path p = this.makeQualified(path);
        switch (PathCapabilitiesSupport.validatePathCapabilityArgs(p, capability)) {
            case "fs.capability.paths.append": 
            case "fs.capability.paths.concat": {
                return false;
            }
        }
        return super.hasPathCapability(p, capability);
    }

    abstract class FsOperation {
        FsOperation() {
        }

        boolean run(Path p) throws IOException {
            Path checkFile;
            boolean status = this.apply(p);
            if (status && ChecksumFileSystem.this.fs.exists(checkFile = ChecksumFileSystem.this.getChecksumFile(p))) {
                this.apply(checkFile);
            }
            return status;
        }

        abstract boolean apply(Path var1) throws IOException;
    }

    private static class ChecksumFSOutputSummer
    extends FSOutputSummer
    implements IOStatisticsSource,
    StreamCapabilities {
        private FSDataOutputStream datas;
        private FSDataOutputStream sums;
        private static final float CHKSUM_AS_FRACTION = 0.01f;
        private boolean isClosed = false;

        ChecksumFSOutputSummer(ChecksumFileSystem fs, Path file, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress, FsPermission permission) throws IOException {
            super(DataChecksum.newDataChecksum(DataChecksum.Type.CRC32, fs.getBytesPerSum()));
            int bytesPerSum = fs.getBytesPerSum();
            this.datas = fs.getRawFileSystem().create(file, permission, overwrite, bufferSize, replication, blockSize, progress);
            int sumBufferSize = fs.getSumBufferSize(bytesPerSum, bufferSize);
            this.sums = fs.getRawFileSystem().create(fs.getChecksumFile(file), permission, true, sumBufferSize, replication, blockSize, null);
            this.sums.write(CHECKSUM_VERSION, 0, CHECKSUM_VERSION.length);
            this.sums.writeInt(bytesPerSum);
        }

        @Override
        public void close() throws IOException {
            try {
                this.flushBuffer();
                this.sums.close();
                this.datas.close();
            }
            finally {
                this.isClosed = true;
            }
        }

        @Override
        protected void writeChunk(byte[] b, int offset, int len, byte[] checksum, int ckoff, int cklen) throws IOException {
            this.datas.write(b, offset, len);
            this.sums.write(checksum, ckoff, cklen);
        }

        @Override
        protected void checkClosed() throws IOException {
            if (this.isClosed) {
                throw new ClosedChannelException();
            }
        }

        @Override
        public IOStatistics getIOStatistics() {
            return IOStatisticsSupport.retrieveIOStatistics(this.datas);
        }

        @Override
        public boolean hasCapability(String capability) {
            if (StoreImplementationUtils.isProbeForSyncable(capability)) {
                return false;
            }
            return this.datas.hasCapability(capability);
        }
    }

    private static class FSDataBoundedInputStream
    extends FSDataInputStream {
        private FileSystem fs;
        private Path file;
        private long fileLen = -1L;

        FSDataBoundedInputStream(FileSystem fs, Path file, InputStream in) {
            super(in);
            this.fs = fs;
            this.file = file;
        }

        @Override
        public boolean markSupported() {
            return false;
        }

        private long getFileLength() throws IOException {
            if (this.fileLen == -1L) {
                this.fileLen = this.fs.getContentSummary(this.file).getLength();
            }
            return this.fileLen;
        }

        @Override
        public synchronized long skip(long n) throws IOException {
            long fileLength;
            long curPos = this.getPos();
            if (n + curPos > (fileLength = this.getFileLength())) {
                n = fileLength - curPos;
            }
            return super.skip(n);
        }

        @Override
        public synchronized void seek(long pos) throws IOException {
            if (pos > this.getFileLength()) {
                throw new EOFException("Cannot seek after EOF");
            }
            super.seek(pos);
        }
    }

    private static class ChecksumFSInputChecker
    extends FSInputChecker
    implements IOStatisticsSource,
    StreamCapabilities {
        private ChecksumFileSystem fs;
        private FSDataInputStream datas;
        private FSDataInputStream sums;
        private static final int HEADER_LENGTH = 8;
        private int bytesPerSum = 1;
        private long fileLen = -1L;

        public ChecksumFSInputChecker(ChecksumFileSystem fs, Path file) throws IOException {
            this(fs, file, fs.getConf().getInt("file.stream-buffer-size", 4096));
        }

        public ChecksumFSInputChecker(ChecksumFileSystem fs, Path file, int bufferSize) throws IOException {
            super(file, fs.getFileStatus(file).getReplication());
            this.datas = fs.getRawFileSystem().open(file, bufferSize);
            this.fs = fs;
            Path sumFile = fs.getChecksumFile(file);
            try {
                int sumBufferSize = fs.getSumBufferSize(fs.getBytesPerSum(), bufferSize);
                this.sums = fs.getRawFileSystem().open(sumFile, sumBufferSize);
                byte[] version = new byte[CHECKSUM_VERSION.length];
                this.sums.readFully(version);
                if (!Arrays.equals(version, CHECKSUM_VERSION)) {
                    throw new IOException("Not a checksum file: " + sumFile);
                }
                this.bytesPerSum = this.sums.readInt();
                this.set(fs.verifyChecksum, DataChecksum.newCrc32(), this.bytesPerSum, 4);
            }
            catch (IOException e) {
                if (!(e instanceof FileNotFoundException) || e.getMessage().endsWith(" (Permission denied)")) {
                    LOG.warn("Problem opening checksum file: " + file + ".  Ignoring exception: ", (Throwable)e);
                }
                this.set(fs.verifyChecksum, null, 1, 0);
            }
        }

        private long getChecksumFilePos(long dataPos) {
            return 8L + 4L * (dataPos / (long)this.bytesPerSum);
        }

        @Override
        protected long getChunkPosition(long dataPos) {
            return dataPos / (long)this.bytesPerSum * (long)this.bytesPerSum;
        }

        @Override
        public int available() throws IOException {
            return this.datas.available() + super.available();
        }

        @Override
        public int read(long position, byte[] b, int off, int len) throws IOException {
            int nread;
            this.validatePositionedReadArgs(position, b, off, len);
            if (len == 0) {
                return 0;
            }
            try (ChecksumFSInputChecker checker = new ChecksumFSInputChecker(this.fs, this.file);){
                checker.seek(position);
                nread = checker.read(b, off, len);
            }
            return nread;
        }

        @Override
        public void close() throws IOException {
            this.datas.close();
            if (this.sums != null) {
                this.sums.close();
            }
            this.set(this.fs.verifyChecksum, null, 1, 0);
        }

        @Override
        public boolean seekToNewSource(long targetPos) throws IOException {
            long sumsPos = this.getChecksumFilePos(targetPos);
            this.fs.reportChecksumFailure(this.file, this.datas, targetPos, this.sums, sumsPos);
            boolean newDataSource = this.datas.seekToNewSource(targetPos);
            return this.sums.seekToNewSource(sumsPos) || newDataSource;
        }

        @Override
        protected int readChunk(long pos, byte[] buf, int offset, int len, byte[] checksum) throws IOException {
            boolean eof = false;
            if (this.needChecksum()) {
                int sumLenRead;
                assert (checksum != null);
                assert (checksum.length % 4 == 0);
                assert (len >= this.bytesPerSum);
                int checksumsToRead = Math.min(len / this.bytesPerSum, checksum.length / 4);
                long checksumPos = this.getChecksumFilePos(pos);
                if (checksumPos != this.sums.getPos()) {
                    this.sums.seek(checksumPos);
                }
                if ((sumLenRead = this.sums.read(checksum, 0, 4 * checksumsToRead)) >= 0 && sumLenRead % 4 != 0) {
                    throw new ChecksumException("Checksum file not a length multiple of checksum size in " + this.file + " at " + pos + " checksumpos: " + checksumPos + " sumLenread: " + sumLenRead, pos);
                }
                if (sumLenRead <= 0) {
                    eof = true;
                } else {
                    len = Math.min(len, this.bytesPerSum * (sumLenRead / 4));
                }
            }
            if (pos != this.datas.getPos()) {
                this.datas.seek(pos);
            }
            int nread = ChecksumFSInputChecker.readFully(this.datas, buf, offset, len);
            if (eof && nread > 0) {
                throw new ChecksumException("Checksum error: " + this.file + " at " + pos, pos);
            }
            return nread;
        }

        @Override
        public IOStatistics getIOStatistics() {
            return IOStatisticsSupport.retrieveIOStatistics(this.datas);
        }

        public static long findChecksumOffset(long dataOffset, int bytesPerSum) {
            return 8L + dataOffset / (long)bytesPerSum * 4L;
        }

        private long getFileLength() throws IOException {
            if (this.fileLen == -1L) {
                this.fileLen = this.fs.getFileStatus(this.file).getLen();
            }
            return this.fileLen;
        }

        public static List<CombinedFileRange> findChecksumRanges(List<? extends FileRange> dataRanges, int bytesPerSum, int minSeek, int maxSize) {
            ArrayList<CombinedFileRange> result = new ArrayList<CombinedFileRange>();
            CombinedFileRange currentCrc = null;
            for (FileRange fileRange : dataRanges) {
                long crcOffset = ChecksumFSInputChecker.findChecksumOffset(fileRange.getOffset(), bytesPerSum);
                long crcEnd = ChecksumFSInputChecker.findChecksumOffset(fileRange.getOffset() + (long)fileRange.getLength() + (long)bytesPerSum - 1L, bytesPerSum);
                if (currentCrc != null && currentCrc.merge(crcOffset, crcEnd, fileRange, minSeek, maxSize)) continue;
                currentCrc = new CombinedFileRange(crcOffset, crcEnd, fileRange);
                result.add(currentCrc);
            }
            return result;
        }

        static ByteBuffer checkBytes(ByteBuffer sumsBytes, long sumsOffset, ByteBuffer data, long dataOffset, int bytesPerSum, Path file) {
            int offset = (int)(ChecksumFSInputChecker.findChecksumOffset(dataOffset, bytesPerSum) - sumsOffset);
            IntBuffer sums = sumsBytes.asIntBuffer();
            sums.position(offset / 4);
            ByteBuffer current = data.duplicate();
            int numFullChunks = data.remaining() / bytesPerSum;
            boolean partialChunk = data.remaining() % bytesPerSum != 0;
            int totalChunks = numFullChunks;
            if (partialChunk) {
                ++totalChunks;
            }
            CRC32 crc = new CRC32();
            for (int c = 0; c < totalChunks; ++c) {
                current.position(c * bytesPerSum);
                if (c == numFullChunks) {
                    int lastIncompleteChunk = data.remaining() % bytesPerSum;
                    current.limit(c * bytesPerSum + lastIncompleteChunk);
                } else {
                    current.limit((c + 1) * bytesPerSum);
                }
                crc.reset();
                crc.update(current);
                int expected = sums.get();
                int calculated = (int)crc.getValue();
                if (calculated == expected) continue;
                long errPosn = dataOffset + (long)c * (long)bytesPerSum;
                throw new CompletionException(new ChecksumException("Checksum error: " + file + " at " + errPosn + " exp: " + expected + " got: " + calculated, errPosn));
            }
            return data;
        }

        private void validateRangeRequest(List<? extends FileRange> ranges, long fileLength) throws EOFException {
            for (FileRange fileRange : ranges) {
                VectoredReadUtils.validateRangeRequest(fileRange);
                if (fileRange.getOffset() + (long)fileRange.getLength() <= fileLength) continue;
                String errMsg = String.format("Requested range [%d, %d) is beyond EOF for path %s", fileRange.getOffset(), fileRange.getLength(), this.file);
                LOG.warn(errMsg);
                throw new EOFException(errMsg);
            }
        }

        @Override
        public void readVectored(List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate) throws IOException {
            long length = this.getFileLength();
            this.validateRangeRequest(ranges, length);
            if (this.sums == null) {
                this.datas.readVectored(ranges, allocate);
                return;
            }
            int minSeek = this.minSeekForVectorReads();
            int maxSize = this.maxReadSizeForVectorReads();
            List<CombinedFileRange> dataRanges = VectoredReadUtils.mergeSortedRanges(Arrays.asList(VectoredReadUtils.sortRanges(ranges)), this.bytesPerSum, minSeek, this.maxReadSizeForVectorReads());
            for (CombinedFileRange range : dataRanges) {
                if (range.getOffset() + (long)range.getLength() <= length) continue;
                range.setLength((int)(length - range.getOffset()));
            }
            List<CombinedFileRange> checksumRanges = ChecksumFSInputChecker.findChecksumRanges(dataRanges, this.bytesPerSum, minSeek, maxSize);
            this.sums.readVectored(checksumRanges, allocate);
            this.datas.readVectored(dataRanges, allocate);
            for (CombinedFileRange checksumRange : checksumRanges) {
                for (FileRange dataRange : checksumRange.getUnderlying()) {
                    CompletionStage result = checksumRange.getData().thenCombineAsync(dataRange.getData(), (sumBuffer, dataBuffer) -> ChecksumFSInputChecker.checkBytes(sumBuffer, checksumRange.getOffset(), dataBuffer, dataRange.getOffset(), this.bytesPerSum, this.file));
                    for (FileRange original : ((CombinedFileRange)dataRange).getUnderlying()) {
                        original.setData((CompletableFuture<ByteBuffer>)((CompletableFuture)result).thenApply(b -> VectoredReadUtils.sliceTo(b, dataRange.getOffset(), original)));
                    }
                }
            }
        }

        @Override
        public boolean hasCapability(String capability) {
            return this.datas.hasCapability(capability);
        }
    }
}

