/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.dashboard;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.amoro.config.Configurations;
import org.apache.amoro.server.AmoroManagementConf;
import org.apache.amoro.server.dashboard.model.OverviewDataSizeItem;
import org.apache.amoro.server.dashboard.model.OverviewResourceUsageItem;
import org.apache.amoro.server.dashboard.model.OverviewTopTableItem;
import org.apache.amoro.server.optimizing.OptimizingStatus;
import org.apache.amoro.server.persistence.PersistentBase;
import org.apache.amoro.server.persistence.TableRuntimeMeta;
import org.apache.amoro.server.persistence.mapper.CatalogMetaMapper;
import org.apache.amoro.server.persistence.mapper.OptimizerMapper;
import org.apache.amoro.server.persistence.mapper.TableMetaMapper;
import org.apache.amoro.server.resource.OptimizerInstance;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.collect.ImmutableList;
import org.apache.amoro.shade.guava32.com.google.common.collect.Maps;
import org.apache.amoro.shade.guava32.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OverviewManager
extends PersistentBase {
    public static final String STATUS_PENDING = "Pending";
    public static final String STATUS_PLANING = "Planing";
    public static final String STATUS_EXECUTING = "Executing";
    public static final String STATUS_IDLE = "Idle";
    public static final String STATUS_COMMITTING = "Committing";
    private static final Logger LOG = LoggerFactory.getLogger(OverviewManager.class);
    private final List<OverviewTopTableItem> allTopTableItem = new ArrayList<OverviewTopTableItem>();
    private final Map<String, Long> optimizingStatusCountMap = new ConcurrentHashMap<String, Long>();
    private final ConcurrentLinkedDeque<OverviewResourceUsageItem> resourceUsageHistory = new ConcurrentLinkedDeque();
    private final ConcurrentLinkedDeque<OverviewDataSizeItem> dataSizeHistory = new ConcurrentLinkedDeque();
    private final AtomicInteger totalCatalog = new AtomicInteger();
    private final AtomicLong totalDataSize = new AtomicLong();
    private final AtomicInteger totalTableCount = new AtomicInteger();
    private final AtomicInteger totalCpu = new AtomicInteger();
    private final AtomicLong totalMemory = new AtomicLong();
    private final int maxRecordCount;

    public OverviewManager(Configurations serverConfigs) {
        this(serverConfigs.getInteger(AmoroManagementConf.OVERVIEW_CACHE_MAX_SIZE), (Duration)serverConfigs.get(AmoroManagementConf.OVERVIEW_CACHE_REFRESH_INTERVAL));
    }

    @VisibleForTesting
    public OverviewManager(int maxRecordCount, Duration refreshInterval) {
        this.maxRecordCount = maxRecordCount;
        ScheduledExecutorService overviewUpdaterScheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("overview-refresh-scheduler-%d").setDaemon(true).build());
        this.resetStatusMap();
        if (refreshInterval.toMillis() > 0L) {
            overviewUpdaterScheduler.scheduleAtFixedRate(this::refresh, 1000L, refreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    public List<OverviewTopTableItem> getAllTopTableItem() {
        return ImmutableList.copyOf(this.allTopTableItem);
    }

    public int getTotalCatalog() {
        return this.totalCatalog.get();
    }

    public int getTotalTableCount() {
        return this.totalTableCount.get();
    }

    public long getTotalDataSize() {
        return this.totalDataSize.get();
    }

    public int getTotalCpu() {
        return this.totalCpu.get();
    }

    public long getTotalMemory() {
        return this.totalMemory.get();
    }

    public List<OverviewResourceUsageItem> getResourceUsageHistory(long startTime) {
        return this.resourceUsageHistory.stream().filter(item -> item.getTs() >= startTime).collect(Collectors.toList());
    }

    public List<OverviewDataSizeItem> getDataSizeHistory(long startTime) {
        return this.dataSizeHistory.stream().filter(item -> item.getTs() >= startTime).collect(Collectors.toList());
    }

    public Map<String, Long> getOptimizingStatus() {
        return this.optimizingStatusCountMap;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void refresh() {
        long start = System.currentTimeMillis();
        LOG.info("Refreshing overview cache");
        try {
            this.refreshTableCache(start);
            this.refreshResourceUsage(start);
        }
        catch (Exception e) {
            LOG.error("Refreshed overview cache failed", (Throwable)e);
        }
        finally {
            long end = System.currentTimeMillis();
            LOG.info("Refreshed overview cache in {} ms.", (Object)(end - start));
        }
    }

    private void refreshTableCache(long ts) {
        int totalCatalogs = this.getAs(CatalogMetaMapper.class, CatalogMetaMapper::selectCatalogCount);
        List metas = this.getAs(TableMetaMapper.class, TableMetaMapper::selectTableRuntimeMetas);
        AtomicLong totalDataSize = new AtomicLong();
        AtomicInteger totalFileCounts = new AtomicInteger();
        HashMap topTableItemMap = Maps.newHashMap();
        HashMap optimizingStatusMap = Maps.newHashMap();
        for (TableRuntimeMeta meta : metas) {
            Optional<OverviewTopTableItem> optItem = this.toTopTableItem(meta);
            optItem.ifPresent(tableItem -> {
                topTableItemMap.put(tableItem.getTableName(), tableItem);
                totalDataSize.addAndGet(tableItem.getTableSize());
                totalFileCounts.addAndGet(tableItem.getFileCount());
            });
            String status = this.statusToMetricString(meta.getTableStatus());
            if (!StringUtils.isNotEmpty((CharSequence)status)) continue;
            optimizingStatusMap.putIfAbsent(status, 0L);
            optimizingStatusMap.computeIfPresent(status, (k, v) -> v + 1L);
        }
        this.totalCatalog.set(totalCatalogs);
        this.totalTableCount.set(topTableItemMap.size());
        this.totalDataSize.set(totalDataSize.get());
        this.allTopTableItem.clear();
        this.allTopTableItem.addAll(topTableItemMap.values());
        this.addAndCheck(new OverviewDataSizeItem(ts, this.totalDataSize.get()));
        this.resetStatusMap();
        this.optimizingStatusCountMap.putAll(optimizingStatusMap);
    }

    private Optional<OverviewTopTableItem> toTopTableItem(TableRuntimeMeta meta) {
        if (meta == null) {
            return Optional.empty();
        }
        OverviewTopTableItem tableItem = new OverviewTopTableItem(this.fullTableName(meta));
        if (meta.getTableSummary() != null) {
            tableItem.setTableSize(meta.getTableSummary().getTotalFileSize());
            tableItem.setFileCount(meta.getTableSummary().getTotalFileCount());
            tableItem.setHealthScore(meta.getTableSummary().getHealthScore());
        }
        tableItem.setAverageFileSize(tableItem.getFileCount() == 0 ? 0L : tableItem.getTableSize() / (long)tableItem.getFileCount());
        return Optional.of(tableItem);
    }

    private String statusToMetricString(OptimizingStatus status) {
        if (status == null) {
            return null;
        }
        switch (status) {
            case PENDING: {
                return STATUS_PENDING;
            }
            case PLANNING: {
                return STATUS_PLANING;
            }
            case MINOR_OPTIMIZING: 
            case MAJOR_OPTIMIZING: 
            case FULL_OPTIMIZING: {
                return STATUS_EXECUTING;
            }
            case IDLE: {
                return STATUS_IDLE;
            }
            case COMMITTING: {
                return STATUS_COMMITTING;
            }
        }
        return null;
    }

    private void resetStatusMap() {
        this.optimizingStatusCountMap.clear();
        this.optimizingStatusCountMap.put(STATUS_PENDING, 0L);
        this.optimizingStatusCountMap.put(STATUS_PLANING, 0L);
        this.optimizingStatusCountMap.put(STATUS_EXECUTING, 0L);
        this.optimizingStatusCountMap.put(STATUS_IDLE, 0L);
        this.optimizingStatusCountMap.put(STATUS_COMMITTING, 0L);
    }

    private void refreshResourceUsage(long ts) {
        List instances = this.getAs(OptimizerMapper.class, OptimizerMapper::selectAll);
        AtomicInteger cpuCount = new AtomicInteger();
        AtomicLong memoryBytes = new AtomicLong();
        for (OptimizerInstance instance : instances) {
            cpuCount.addAndGet(instance.getThreadCount());
            memoryBytes.addAndGet((long)instance.getMemoryMb() * 1024L * 1024L);
        }
        this.totalCpu.set(cpuCount.get());
        this.totalMemory.set(memoryBytes.get());
        this.addAndCheck(new OverviewResourceUsageItem(ts, cpuCount.get(), memoryBytes.get()));
    }

    private void addAndCheck(OverviewDataSizeItem dataSizeItem) {
        this.dataSizeHistory.add(dataSizeItem);
        this.checkSize(this.dataSizeHistory);
    }

    private void addAndCheck(OverviewResourceUsageItem resourceUsageItem) {
        this.resourceUsageHistory.add(resourceUsageItem);
        this.checkSize(this.resourceUsageHistory);
    }

    private <T> void checkSize(Deque<T> deque) {
        if (deque.size() > this.maxRecordCount) {
            deque.poll();
        }
    }

    private String fullTableName(TableRuntimeMeta meta) {
        return meta.getCatalogName().concat(".").concat(meta.getDbName()).concat(".").concat(meta.getTableName());
    }
}

